package message

import (
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
)

const (
	moveScopesOffset = headerLength
)

// Move removes scopes from the available list originally reported by the broker
type Move interface {
	protocol.Message
	Scopes() stream.AddressScopes
}

type moveMessage struct {
	scopes stream.AddressScopes
}

// NewMove creates a Move
func NewMove(scopes stream.AddressScopes) (Move, error) {
	if len(scopes) == 0 {
		return nil, protocol.ErrInvalidAddress
	}
	return &moveMessage{scopes}, nil
}

func (*moveMessage) OpCode() protocol.OpCode        { return protocol.Move }
func (m *moveMessage) Scopes() stream.AddressScopes { return m.scopes }

func (m *moveMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if ver != protocol.Five {
		return nil, protocol.ErrInvalidVersion
	}
	scopes, _ := m.scopes.MarshalBinary()
	bytes := make([]byte, moveScopesOffset+len(scopes))
	injectHeader(bytes, ver, m.OpCode())
	copy(bytes[moveScopesOffset:], scopes)
	return bytes, nil
}

func (m *moveMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if ver != protocol.Five {
		return protocol.ErrInvalidVersion
	}
	if len(bytes) < moveScopesOffset {
		return protocol.ErrInvalidLength(moveScopesOffset, len(bytes))
	}
	var scopes stream.AddressScopes
	if err := scopes.UnmarshalBinary(bytes[moveScopesOffset:]); err != nil {
		return err
	}
	m.scopes = scopes
	return nil
}

func (m *moveMessage) String() string {
	return fmt.Sprintf("<%v> %v", m.OpCode(), m.Scopes())
}
