package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
)

const (
	partRequestIDOffset = headerLength
	partAddressOffset   = partRequestIDOffset + 2
)

// Part attempts to unsubscribe from a stream address
type Part interface {
	protocol.Request
	Address() stream.Address
}

type partMessage struct {
	id   protocol.RequestID
	addr stream.Address
}

// NewPart creates a Part
func NewPart(id protocol.RequestID, addr stream.Address) (Part, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidRequestID
	}
	if addr == nil {
		return nil, protocol.ErrInvalidAddress
	}
	return &partMessage{id, addr}, nil
}

func (*partMessage) OpCode() protocol.OpCode         { return protocol.Part }
func (p *partMessage) RequestID() protocol.RequestID { return p.id }
func (p *partMessage) Address() stream.Address       { return p.addr }
func (p *partMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	key := stream.NoAddress
	if p.addr != nil {
		key = p.addr.Key()
	}
	bytes := make([]byte, partAddressOffset+len(key))
	injectHeader(bytes, ver, p.OpCode())
	binary.BigEndian.PutUint16(bytes[partRequestIDOffset:], uint16(p.id))
	copy(bytes[partAddressOffset:], key)
	return bytes, nil
}

func (p *partMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	if len(bytes) < partAddressOffset {
		return protocol.ErrInvalidLength(partAddressOffset, len(bytes))
	}

	id := protocol.RequestID(binary.BigEndian.Uint16(bytes[partRequestIDOffset:]))
	if !id.IsValid() {
		return protocol.ErrInvalidRequestID
	}

	addr, err := stream.ParseAddress(string(bytes[partAddressOffset:]))
	if err != nil {
		return err
	}
	p.id = id
	p.addr = addr
	return nil
}

func (p *partMessage) String() string {
	return fmt.Sprintf("<%v> %d %s", p.OpCode(), p.RequestID(), p.Address().Key())
}
