package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
)

const (
	releaseRequestIDOffset = headerLength
	releaseAddressOffset   = releaseRequestIDOffset + 2
)

// Release attempts to unsubscribe from a stream address
type Release interface {
	protocol.Request
	Address() stream.Address
}

type releaseMessage struct {
	id   protocol.RequestID
	addr stream.Address
}

// NewRelease creates a Release
func NewRelease(id protocol.RequestID, addr stream.Address) (Release, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidRequestID
	}
	if addr == nil {
		return nil, protocol.ErrInvalidAddress
	}
	return &releaseMessage{id, addr}, nil
}

func (*releaseMessage) OpCode() protocol.OpCode         { return protocol.Release }
func (p *releaseMessage) RequestID() protocol.RequestID { return p.id }
func (p *releaseMessage) Address() stream.Address       { return p.addr }
func (p *releaseMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	key := stream.NoAddress
	if p.addr != nil {
		key = p.addr.Key()
	}
	bytes := make([]byte, releaseAddressOffset+len(key))
	injectHeader(bytes, ver, p.OpCode())
	binary.BigEndian.PutUint16(bytes[releaseRequestIDOffset:], uint16(p.id))
	copy(bytes[releaseAddressOffset:], key)
	return bytes, nil
}

func (p *releaseMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	if len(bytes) < releaseAddressOffset {
		return protocol.ErrInvalidLength(releaseAddressOffset, len(bytes))
	}

	id := protocol.RequestID(binary.BigEndian.Uint16(bytes[releaseRequestIDOffset:]))
	if !id.IsValid() {
		return protocol.ErrInvalidRequestID
	}

	addr, err := stream.ParseAddress(string(bytes[releaseAddressOffset:]))
	if err != nil {
		return err
	}
	p.id = id
	p.addr = addr
	return nil
}

func (p *releaseMessage) String() string {
	return fmt.Sprintf("<%v> %d %s", p.OpCode(), p.RequestID(), p.Address().Key())
}
