package registry

import (
	"sort"

	"code.justin.tv/devhub/e2ml/libs/stream"
)

type addressSet struct {
	binding *binding
	scopes  stream.AddressScopes
	addrs   map[stream.AddressKey]stream.Address
}

func newAddressSet(binding *binding, s stream.AddressScopes) *addressSet {
	scopes := make(stream.AddressScopes, len(s))
	copy(scopes, s)
	sort.Sort(scopes)
	return &addressSet{
		addrs:   make(map[stream.AddressKey]stream.Address),
		binding: binding,
		scopes:  scopes,
	}
}

func (a *addressSet) find(addr stream.Address, minCardinality int) (int, bool) {
	return a.scopes.HasBetterMatch(addr, minCardinality)
}

func (a *addressSet) insert(addr stream.Address) {
	a.addrs[addr.Key()] = addr
}

func (a *addressSet) addScopes(scopes stream.AddressScopes) {
	union := make(stream.AddressScopes, len(a.scopes))
	copy(union, a.scopes)
	for _, s := range scopes {
		if union.Contains(s) {
			continue
		}
		union = append(union, s)
	}
	sort.Sort(union)
	a.scopes = union
}

func (a *addressSet) dropScopes(scopes stream.AddressScopes) []stream.Address {
	filtered := stream.AddressScopes{}
	for _, s := range a.scopes {
		if scopes.Contains(s) {
			continue
		}
		filtered = append(filtered, s)
	}
	sort.Sort(filtered)
	remapped := make(map[stream.AddressKey]stream.Address)
	culled := []stream.Address{}
	for key, addr := range a.addrs {
		if _, found := filtered.HasBetterMatch(addr, stream.NoAddressMatch); found {
			remapped[key] = addr
		} else {
			culled = append(culled, addr)
		}
	}
	a.scopes = filtered
	a.addrs = remapped
	return culled
}

func (a *addressSet) remove(addr stream.Address) bool {
	delete(a.addrs, addr.Key())
	return len(a.addrs) == 0
}

func (a *addressSet) list() []stream.Address {
	addrs := make([]stream.Address, 0, len(a.addrs))
	for _, addr := range a.addrs {
		addrs = append(addrs, addr)
	}
	return addrs
}
