package registry

import (
	"sync"
	"time"

	"code.justin.tv/devhub/lib-lifecycle/src/lifecycle"
)

const (
	retryStep      = 500 * time.Millisecond
	maxRetryDelay  = 7500 * time.Millisecond
	retryThreshold = maxRetryDelay + retryStep
)

type retryLogic struct {
	lastAttempt time.Time
	nextDelay   time.Duration
	onCancel    lifecycle.Manager
	active      bool
	mutex       sync.Mutex
}

func newRetryLogic(onCancel lifecycle.Manager) *retryLogic {
	return &retryLogic{onCancel: onCancel}
}

func (r *retryLogic) schedule(task func()) {
	now := time.Now()
	r.mutex.Lock()
	defer r.mutex.Unlock()
	if r.active {
		return
	}
	r.active = true
	delta := now.Sub(r.lastAttempt)
	delay := currentDelay(delta, r.nextDelay)
	r.nextDelay = nextDelay(delay)
	timer := time.AfterFunc(delay, r.doTask(task))
	r.onCancel.RegisterHook(r, r.doCancel(timer))
}

func (r *retryLogic) doTask(task func()) func() {
	return func() {
		r.mutex.Lock()
		r.active = false
		r.lastAttempt = time.Now()
		r.mutex.Unlock()
		task()
	}
}

func (r *retryLogic) doCancel(timer *time.Timer) func() error {
	return func() error {
		timer.Stop()
		r.mutex.Lock()
		r.active = false
		r.mutex.Unlock()
		return nil
	}
}

func (r *retryLogic) cancel() error { return r.onCancel.ExecuteHook(r) }

func currentDelay(elapsed time.Duration, expected time.Duration) time.Duration {
	current := expected
	if elapsed > retryThreshold {
		current = 0
	}
	return current
}

func nextDelay(delay time.Duration) time.Duration {
	next := 2*delay + retryStep
	if next > maxRetryDelay {
		next = maxRetryDelay
	}
	return next
}
