package stream

import "sync/atomic"

// Tracker holds the last known source and position for a given stream; when it
// is updated, it will return false if the update should be ignored due to being
// a repeated signal or out of sequence. This is accomplished by bit packing the
// two 32-bit values into a 64-bit uint with the sequence in the high bits for
// simplified comparison mechanics.
type Tracker uint64

const (
	Untracked Tracker = 0
)

func CreateTracker(src SourceID, pos Position) Tracker {
	return Tracker(uint64(pos)<<32 | uint64(src))
}

func (t *Tracker) Current() (SourceID, Position) {
	if t == nil {
		return None, Origin
	}
	data := atomic.LoadUint64((*uint64)(t))
	return SourceID(data), Position(data >> 32)
}

func (t *Tracker) Next() (SourceID, Position) {
	if t == nil {
		return None, Origin
	}
	data := atomic.AddUint64((*uint64)(t), 1<<32)
	return SourceID(data), Position(data >> 32)
}

func (t *Tracker) SetSource(src SourceID) bool { return t.Set(src, Origin) }

func (t *Tracker) Set(src SourceID, pos Position) bool {
	if t == nil {
		return false
	}
	for {
		current := t.toData(src, pos)
		prev := atomic.LoadUint64((*uint64)(t))
		if current <= prev && SourceID(prev) == src {
			return false
		}
		if atomic.CompareAndSwapUint64((*uint64)(t), prev, current) {
			return true
		}
	}
}

func (t *Tracker) SetPosition(pos Position) (SourceID, bool) {
	if t == nil {
		return None, false
	}
	for {
		prev := atomic.LoadUint64((*uint64)(t))
		source := SourceID(prev)
		current := t.toData(source, pos)
		if current <= prev {
			return None, false
		}
		if atomic.CompareAndSwapUint64((*uint64)(t), prev, current) {
			return source, true
		}
	}
}

func (t *Tracker) Accept(desc MessageDescription) bool {
	return t.Set(desc.Source(), desc.At().End)
}

func (*Tracker) toData(src SourceID, pos Position) uint64 {
	return uint64(src) | uint64(pos)<<32
}
