package lifecycle

import (
	"sync"
)

// Reporter can process any errors caused by shutdown hooks
type Reporter func(interface{}, error)

var (
	reporterMutex sync.RWMutex

	nilReporter          Reporter = func(interface{}, error) {}
	defaultErrorReporter          = nilReporter
	defaultPanicReporter          = nilReporter

	proxiedDefaultErrorReporter = func(val interface{}, err error) {
		reporterMutex.RLock()
		rep := defaultErrorReporter
		reporterMutex.RUnlock()
		rep(val, err)
	}

	proxiedDefaultPanicReporter = func(val interface{}, err error) {
		reporterMutex.RLock()
		rep := defaultPanicReporter
		reporterMutex.RUnlock()
		rep(val, err)
	}
)

func SetDefaultErrorReporter(reporter Reporter) {
	reporterMutex.Lock()
	defaultErrorReporter = reporter
	reporterMutex.Unlock()
}

func SetDefaultPanicReporter(reporter Reporter) {
	reporterMutex.Lock()
	defaultPanicReporter = reporter
	reporterMutex.Unlock()
}
