package main

import (
	"encoding/json"
	"flag"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"os"
	"strconv"
	"time"

	httpx "code.justin.tv/devhub/e2ml/libs/http"
)

const (
	EnvOauth      = "TWITCH_OAUTH"
	maxBodyLength = 1024 * 10
)

type ExtensionPermissionData struct {
	ExtensionID string          `json:"extension_id"`
	Token       string          `json:"token"`
	Grants      map[string]bool `json:"grants"`
}

// swagger:model GetExtensionsByChannelResponse
type ExtensionsByChannel struct {
	IssuedAt time.Time                  `json:"issued_at"`
	Tokens   []*ExtensionPermissionData `json:"tokens"`
}

func checkEnv(key string) *string {
	if value, ok := os.LookupEnv(key); ok {
		return &value
	}
	return nil
}
func main() {
	channelID := flag.String("channel-id", "", "channel to use")
	extensionID := flag.String("extension-id", "", "extension to connect to")
	env := flag.String("env", "staging", "determines which env the token will be vailid for [staging,prod]")
	flag.Parse()
	oauth := checkEnv(EnvOauth)
	visageURL := getVisageURL(*env)
	exts, err := makeRequest(oauth, visageURL, *channelID)
	if err != nil {
		panic("No jwt for requested extension")
	}
	var jwt string
	for _, ext := range exts.Tokens {
		if ext.ExtensionID == *extensionID {
			jwt = ext.Token
			// fmt.Println(jwt)
		}
	}
	if jwt == "" {
		panic("No jwt for requested extension")
	}
	fmt.Println("export TWITCH_EXT_JWT='" + jwt + "'")
	fmt.Println("export TWITCH_EXT_ID='" + *extensionID + "'")

}

func makeRequest(oauth *string, visageURL string, channelID string) (*ExtensionsByChannel, error) {
	reqUrl, err := url.Parse(visageURL)
	if err != nil {
		return nil, err
	}

	if _, err := strconv.ParseUint(channelID, 10, 64); err != nil {
		return nil, err
	}

	reqUrl.Path = "/v5/channels/" + channelID + "/extensions"

	req, err := http.NewRequest("GET", reqUrl.String(), nil)
	if err != nil {
		return nil, err
	}

	req.Header.Add("Client-ID", "jzkbprff40iqj646a697cyrvl0zt2m6")
	if oauth != nil {
		req.Header.Add("authorization", "OAuth "+*oauth)
	}

	var out ExtensionsByChannel
	if err = execute(req, &out); err != nil {
		return nil, err
	}
	return &out, nil
}

func getVisageURL(env string) string {
	switch env {
	case "staging":
		return "https://visage-beta.twitch.tv"
	case "prod":
		return "https://api.twitch.tv"
	}
	panic("Unknown environment")
}

func execute(req *http.Request, out interface{}) error {
	client := http.Client{}
	resp, err := client.Do(req)
	if err != nil {
		return err
	}

	defer resp.Body.Close()
	if httpErr, found := httpx.ExtractErrorFromResponse(resp); found {
		return httpErr
	}

	if err = json.NewDecoder(io.LimitReader(resp.Body, maxBodyLength)).Decode(out); err != nil {
		return fmt.Errorf("Unable to read response body: %s", err)
	}
	return nil
}
