package main

import (
	"bufio"
	"crypto/tls"
	"fmt"
	"net/url"
	"os"
	"time"

	"code.justin.tv/devhub/lib-lifecycle/src/lifecycle"
	"code.justin.tv/devhub/e2ml/clients/chat"
	"code.justin.tv/devhub/e2ml/clients/chat/setup"
	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/timeout"
	"code.justin.tv/devhub/e2ml/libs/websocket"
	"github.com/alexflint/go-arg"
)

func main() {
	var config setup.Config
	arg.MustParse(&config)

	console := logging.NewConsole(config.Log)
	logger := console.Log

	mgr := lifecycle.NewManager()
	defer func() { _ = mgr.ExecuteAll() }()
	lifecycle.SetDefaultPanicReporter(func(key interface{}, err error) { logger(logging.Error, err) })
	lifecycle.SetDefaultErrorReporter(func(key interface{}, err error) {
		logger(logging.Error, fmt.Sprintf("Shutdown error for %+v: %v", key, err))
	})

	settings := &websocket.Settings{
		Certs:     &tls.Config{InsecureSkipVerify: config.Insecure},
		Lifecycle: mgr,
		Logger:    logger,
		Timeout:   timeout.NewConstantSampler(20 * time.Second),
	}

	url := url.URL(config.Server)
	state, err := chat.NewState(
		string(config.Username),
		setup.NewBaseAddress(config.ClientID, config.JWT),
		console,
		setup.NewAuthSource(config.ClientID, config.JWT, config.S2SSecret, config.JWTDuration, logger),
		setup.NewBroker(&url, settings.Certs, logger),
		setup.NewMetricsTracker(config.MetricsMethod, config, logger),
		websocket.NewClientResolver(settings),
	)

	if err == nil {
		mgr.RegisterHook(state, func() error {
			state.Close()
			return nil
		})
		logger(logging.Info, "# Using name", config.Username, "to chat")
		logger(logging.Info, "# Type \"/join <channel name>\" to join a channel")
		logger(logging.Info, "# Type any text after joining to send messages")
		err = state.Run()
	}
	if err != nil {
		errBuffer := bufio.NewWriter(os.Stderr)
		_, _ = errBuffer.WriteString(err.Error())
		errBuffer.Flush()
	}
}
