package chat

import (
	"encoding/json"
	"sync"

	"code.justin.tv/devhub/e2ml/libs/stream"
)

const (
	topicFilter = "t"
)

type listener struct {
	format   *formatter
	trackers map[stream.AddressKey]*stream.Tracker
	mutex    sync.RWMutex
}

func newListener(format *formatter) stream.Listener {
	return &listener{format: format, trackers: make(map[stream.AddressKey]*stream.Tracker)}
}

func (l *listener) Current(addr stream.Address) (stream.SourceID, stream.Position) {
	l.mutex.RLock()
	pos := l.trackers[addr.Key()]
	l.mutex.RUnlock()
	return pos.Current()
}

func (l *listener) OnDataReceived(msg stream.Message) bool {
	if !l.advance(msg) { // duplicate message, ignore it
		return true
	}
	content := new(message)
	if err := json.Unmarshal(msg.Data(), content); err != nil {
		return false
	}
	if topic, ok := msg.Address().Filter(topicFilter); ok {
		l.format.onMessage(topic, content)
		return true
	}
	return false
}

func (l *listener) OnDataLost(desc stream.MessageDescription) bool {
	l.advance(desc)
	return true
}

func (l *listener) OnStreamClosed(addr stream.Address, err error) bool {
	l.mutex.Lock()
	delete(l.trackers, addr.Key())
	l.mutex.Unlock()
	if topic, ok := addr.Filter(topicFilter); ok {
		l.format.onClosed(topic)
	}
	return true
}

func (l *listener) advance(desc stream.MessageDescription) bool {
	key := desc.Address().Key()
	l.mutex.Lock()
	tracker, ok := l.trackers[key]
	if !ok {
		tracker = new(stream.Tracker)
		l.trackers[key] = tracker
	}
	l.mutex.Unlock()
	return tracker.Accept(desc)
}
