package chat

import (
	"errors"
	"fmt"
	"strings"

	"code.justin.tv/devhub/e2ml/libs/logging"
)

type command func(args []string) (bool, error)

func adapter(cmd func(arg string) error) func(args []string) (bool, error) {
	return func(args []string) (bool, error) {
		if len(args) != 1 {
			return false, fmt.Errorf("Expected one argument")
		}
		return false, cmd(args[0])
	}
}

type parser struct {
	console  logging.Console
	commands map[string]command
	onError  func(error)
}

func newParser(console logging.Console, cmds map[string]command, onError func(error)) *parser {
	return &parser{console: console, commands: cmds, onError: onError}
}

func (p *parser) run() {
	var quit bool
	var msg string
	var err error
	for msg, err = p.console.ReadLine(); err == nil; msg, err = p.console.ReadLine() {
		if len(msg) > 0 {
			quit, err = p.parse(msg)
			if err != nil {
				p.onError(err)
			}
			if quit {
				break
			}
		}
	}
}

func (p *parser) parse(msg string) (bool, error) {
	if strings.HasPrefix(msg, "/") {
		segments := strings.Fields(msg)
		cmd := segments[0][1:]
		args := segments[1:]
		if command, ok := p.commands[cmd]; ok {
			return command(args)
		}
		return false, fmt.Errorf("Unknown command \"%s\"", cmd)
	}
	if command, ok := p.commands[cmdSend]; ok {
		return command([]string{msg})
	}
	return false, errors.New("Send command not found, unable to send message")
}
