package chat

import (
	"encoding/json"
	"errors"
	"fmt"
	"sync"
	"time"

	"code.justin.tv/devhub/e2ml/libs/discovery"
	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/metrics"
	"code.justin.tv/devhub/e2ml/libs/session"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/registry"
)

const (
	cmdJoin  = "join"
	cmdLeave = "leave"
	cmdLog   = "log"
	cmdQuit  = "quit"
	cmdSend  = "sent"
)

type topicMap map[string]stream.Writer

type state struct {
	username  string
	baseAddr  stream.Address
	current   string
	topics    topicMap
	console   logging.Console
	formatter *formatter
	parser    *parser
	listener  stream.Listener
	reg       registry.Remote
	mutex     sync.RWMutex
}

func NewState(
	username string,
	addr stream.Address,
	console logging.Console,
	auth stream.AuthSource,
	broker discovery.Broker,
	stats metrics.Tracker,
	resolver session.ClientResolver,
) (*state, error) {
	formatter := &formatter{console}
	s := &state{
		username:  username,
		baseAddr:  addr,
		reg:       registry.NewRemote(auth, broker, resolver, time.Second, stats, console.Log),
		listener:  newListener(formatter),
		console:   console,
		formatter: formatter,
		topics:    make(topicMap),
	}
	s.parser = newParser(console, map[string]command{
		cmdLog:   adapter(s.logLevel),
		cmdJoin:  adapter(s.tryJoin),
		cmdQuit:  s.quit,
		cmdSend:  adapter(s.talk),
		cmdLeave: adapter(s.tryLeave),
	}, formatter.onError)

	return s, nil
}

func (s *state) Run() error {
	s.parser.run()
	return nil
}

func (s *state) Close() error {
	s.quit([]string{})
	s.reg.Close()
	return s.console.Close()
}

func (s *state) String() string { return s.username }

func (s *state) logLevel(arg string) error {
	level, ok := logging.ParseLevel(arg)
	if !ok {
		return errors.New("invalid log level: " + arg)
	}
	s.console.SetLevel(level)
	s.formatter.onLogLevelSet(level)
	return nil
}

func (s *state) tryLeave(topic string) error {
	addr, err := s.baseAddr.WithFilter(topicFilter, topic)
	if err != nil {
		return err
	}

	s.mutex.Lock()
	writer, ok := s.topics[topic]
	delete(s.topics, topic)
	if s.current == topic {
		s.current = ""
		for t := range s.topics { // grab first if it exists
			s.current = t
			break
		}
	}
	s.mutex.Unlock()
	s.printTopics()
	if ok {
		go func() {
			_, rerr := s.reg.Reader(addr).Leave(s.listener).Result()
			if rerr != nil {
				s.formatter.onError(rerr)
			}
		}()
		go func() {
			_, rerr := s.send(writer, topic, &message{Code: "leave", User: s.username}).Result()
			if rerr != nil {
				s.formatter.onError(rerr)
			}
			rerr = writer.Close()
			if rerr != nil {
				s.formatter.onError(rerr)
			}
		}()
	}
	return nil
}

func (s *state) tryJoin(topic string) error {
	addr, err := s.baseAddr.WithFilter(topicFilter, topic)
	if err != nil {
		return err
	}
	s.mutex.Lock()
	_, ok := s.topics[topic]
	if !ok {
		s.topics[topic] = s.reg.Writer(addr)
	}
	s.current = topic
	s.mutex.Unlock()
	if ok {
		s.printTopics()
		return nil
	}
	if _, err := s.reg.Reader(addr).Join(s.listener).Result(); err != nil {
		s.console.WriteError(err)
		return s.tryLeave(topic)
	}
	s.printTopics()
	return s.trySend(topic, &message{Code: "join", User: s.username})
}

func (s *state) printTopics() {
	s.mutex.RLock()
	current := s.current
	topics := make([]string, 0, len(s.topics))
	for topic := range s.topics {
		topics = append(topics, topic)
	}
	s.mutex.RUnlock()
	s.formatter.printTopics(current, topics)
}

func (s *state) talk(msg string) error {
	s.mutex.RLock()
	topic := s.current
	s.mutex.RUnlock()
	if topic == "" {
		return fmt.Errorf("No current channel")
	}
	return s.trySend(topic, &message{Code: "send", User: s.username, Content: msg})
}

func (s *state) quit([]string) (bool, error) {
	s.mutex.Lock()
	topics := s.topics
	s.topics = make(topicMap)
	s.mutex.Unlock()
	for topic := range topics {
		s.tryLeave(topic)
	}
	return true, nil
}

func (s *state) send(writer stream.Writer, topic string, msg *message) stream.TrackerPromise {
	bytes, _ := json.Marshal(msg)
	return writer.Send(bytes, true)
}

func (s *state) trySend(topic string, msg *message) error {
	s.mutex.RLock()
	writer, ok := s.topics[topic]
	s.mutex.RUnlock()
	if !ok {
		return nil
	}
	go func() {
		if _, err := s.send(writer, topic, msg).Result(); err != nil {
			s.formatter.onError(err)
		}
	}()
	return nil
}
