package main

import (
	"bufio"
	"crypto/tls"
	"fmt"
	"os"
	"time"

	"github.com/alexflint/go-arg"

	identifier "code.justin.tv/amzn/TwitchProcessIdentifier"
	"code.justin.tv/devhub/e2ml/clients/collider"
	"code.justin.tv/devhub/e2ml/libs/discovery/host"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/metrics"
	"code.justin.tv/devhub/e2ml/libs/setup"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/auth/fake"
	"code.justin.tv/devhub/e2ml/libs/timeout"
	"code.justin.tv/devhub/e2ml/libs/websocket"
	"code.justin.tv/devhub/lib-lifecycle/src/lifecycle"
)

type Config struct {
	metrics.GoArgSource

	HostURL        setup.URL     `arg:"--host-url,env:HOST_URL" default:"wss://greeter:3010" help:"url for host connection"`
	HostName       setup.URL     `arg:"--host-name,env:HOST_NAME" default:"ws://threshold:3002" help:"host name this instance will report to discovery"`
	HostHeartbeat  time.Duration `arg:"--host-heartbest,env:HOST_HEARTBEAT" default:"10s" help:"minimum time before status reports to the host"`
	HostWindowSize int           `arg:"--host-window-size,env:HOST_WINDOW_SIZE" default:"40" help:"number of samples used to track host load"`
	Log            logging.Level `default:"info" help:"logging level [trace,debug,info,warning,error]"`
	MetricsHost    string        `arg:"--metrics-host,env:METRICS_HOST" default:"localhost" help:"hostname for metric reporting"`
	MetricsMethod  string        `arg:"--metrics-method,env:METRICS_METHOD" default:"none" help:"method for metric collection [none,logged]"`
	MetricsPeriod  time.Duration `arg:"--metrics-period,env:METRICS_PERIOD" default:"10s" help:"period between metrics reporting ticks"`
	Insecure       bool          `help:"allow connection to insecure server; only for local testing"`
}

func main() {
	var config Config
	arg.MustParse(&config)

	console := logging.NewConsole(config.Log)
	logger := console.Log

	mgr := lifecycle.NewManager()
	defer func() { _ = mgr.ExecuteAll() }()
	lifecycle.SetDefaultPanicReporter(func(key interface{}, err error) { logger(logging.Error, err) })
	lifecycle.SetDefaultErrorReporter(func(key interface{}, err error) {
		logger(logging.Error, fmt.Sprintf("Shutdown error for %+v: %v", key, err))
	})

	metrics := setup.MetricsTracker(config.MetricsMethod, identifier.ProcessIdentifier{}, logger)
	mgr.RegisterHook(metrics, func() error {
		err := metrics.Close()
		time.Sleep(time.Second) // allow metrics to drain
		return err
	})
	mgr.TickUntilClosed(metrics.Tick, config.MetricsPeriod)

	hostClients := websocket.NewClientFactory(setup.Unwrap(config.HostURL), &websocket.Settings{
		Certs:     &tls.Config{InsecureSkipVerify: config.Insecure},
		Lifecycle: mgr,
		Logger:    logger,
		Timeout:   timeout.NewConstantSampler(20 * time.Second),
	})

	reporter, err := host.NewReporter(
		hostClients,
		setup.Unwrap(config.HostName),
		stream.AddressSourceMap{stream.AnyAddress.Key(): stream.None},
		collider.Store(),
		onAddressRequest,
		fake.NewAuthSource(),
		metrics,
		console.Log,
	)
	if err != nil {
		setup.PanicWithMessage("Unable to start reporter", err)
	}
	mgr.RegisterHook(reporter, reporter.Close)

	status := host.NewRateLimitedStatus(reporter, protocol.Available|protocol.Source)
	mgr.RegisterHook(status, status.Close)

	state, err := collider.NewState(
		console,
		reporter,
	)
	if err == nil {
		mgr.RegisterHook(state, state.Close)
		status.Start(state.LoadFactor, config.HostWindowSize)
		reporter.SetSourceLogic(state)
		reporter.UpdateStatus(protocol.NoLoad, protocol.Available|protocol.Source)
		mgr.TickUntilClosed(status.Tick, config.HostHeartbeat)

		logger(logging.Info, "# Type \"/add <address> <sourceID>\" to add an address. Check Source logs to see what addr and sourceID to use to cause collisions.")
		logger(logging.Info, "# Type \"/drop <address>\" to drop an address")
		logger(logging.Info, "# Type \"/load <#>\" to set simulated service load; use 0 for a random factor")
		logger(logging.Info, "# Type \"/quit\" to end the program")
		err = state.Run()
	}

	if err != nil {
		errBuffer := bufio.NewWriter(os.Stderr)
		_, _ = errBuffer.WriteString(err.Error())
		errBuffer.Flush()
	}
}

func onAddressRequest(addr stream.Address, creds stream.Credentials) (stream.AddressScopes, error) {
	return nil, stream.ErrNotImplemented
}
