package collider

import (
	"fmt"
	"math/rand"
	"strconv"
	"sync"
	"sync/atomic"
	"time"

	"code.justin.tv/devhub/e2ml/libs/discovery"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

const (
	cmdAdd  = "add"
	cmdDrop = "drop"
	cmdLoad = "load"
	cmdQuit = "quit"
)

func init() {
	rand.Seed(time.Now().Unix())
}

type topicMap map[string]stream.AddressScope

type state struct {
	topics    topicMap
	console   logging.Console
	reporter  discovery.HostReporter
	formatter *formatter
	parser    *parser
	entries   stream.AddressSourceMap
	load      uint64
	mutex     sync.RWMutex
}

var _ discovery.SourceLogic = (*state)(nil)

func NewState(
	console logging.Console,
	reporter discovery.HostReporter,
) (*state, error) {
	formatter := &formatter{console}
	s := &state{
		console:   console,
		formatter: formatter,
		topics:    make(topicMap),
		reporter:  reporter,
		entries:   make(stream.AddressSourceMap),
	}
	s.parser = newParser(console, map[string]command{
		cmdAdd:  s.add,
		cmdDrop: s.drop,
		cmdLoad: s.setLoad,
		cmdQuit: s.quit,
	}, formatter.onError)
	return s, nil
}

func (s *state) Run() error {
	s.parser.run()
	return nil
}

func (s *state) Close() error {
	s.quit([]string{})
	return s.console.Close()
}

func (s *state) LoadFactor() uint64 {
	load := atomic.LoadUint64(&s.load)
	if load == 0 {
		load = uint64((rand.Float32()*0.2 + 0.8) * protocol.MaxLoad / 4)
	}
	return load
}

func (s *state) setLoad(args []string) (bool, error) {
	if len(args) != 1 {
		return false, fmt.Errorf("Expected one argument")
	}
	load, err := strconv.ParseUint(args[0], 10, 64)
	if err != nil {
		return false, err
	}
	atomic.StoreUint64(&s.load, load)
	return false, nil
}

func (s *state) add(args []string) (bool, error) {
	if len(args) != 2 {
		return false, fmt.Errorf("Expected two arguments")
	}
	id, err := strconv.ParseUint(args[1], 10, 32)
	if err != nil {
		return false, err
	}
	return s.insertEntry(stream.AddressKey(args[0]), stream.SourceID(id))
}

func (s *state) drop(args []string) (bool, error) {
	if len(args) != 1 {
		return false, fmt.Errorf("Expected one argument")
	}
	return s.removeEntry(stream.AddressKey(args[0]))
}

func (s *state) ServeAddress(addr stream.Address) (stream.AddressScopes, error) {
	id := stream.SourceID(rand.Int31())
	if _, err := s.insertEntry(addr.Key(), id); err != nil {
		return nil, err
	}
	return stream.AddressScopes{addr}, nil
}

func (s *state) RevokeAddress(addr stream.Address) (stream.AddressScopes, error) {
	if _, err := s.removeEntry(addr.Key()); err != nil {
		return nil, err
	}
	return stream.AddressScopes{addr}, nil
}

func (s *state) insertEntry(key stream.AddressKey, src stream.SourceID) (bool, error) {
	scope, err := key.ToAddressScope()
	if err != nil {
		return false, err
	}
	s.mutex.Lock()
	s.entries[key] = src
	s.mutex.Unlock()
	go s.reporter.AddSupported(stream.AddressSourceMap{key: src})
	s.formatter.onAdded(scope, src)
	return false, nil
}

func (s *state) removeEntry(key stream.AddressKey) (bool, error) {
	scope, err := key.ToAddressScope()
	if err != nil {
		return false, err
	}
	s.mutex.Lock()
	src, found := s.entries[key]
	delete(s.entries, key)
	s.mutex.Unlock()
	if !found {
		return false, nil
	}
	go s.reporter.DropSupported(stream.AddressSourceMap{key: src})
	s.formatter.onDropped(scope)
	return false, nil
}

func (s *state) quit([]string) (bool, error) {
	s.mutex.Lock()
	topics := s.topics
	s.topics = make(topicMap)
	s.mutex.Unlock()
	for _, scope := range topics {
		s.drop([]string{string(scope.Key())})
	}
	return true, nil
}
