package main

import (
	"crypto/tls"
	"time"

	"code.justin.tv/devhub/lib-lifecycle/src/lifecycle"
	"code.justin.tv/devhub/e2ml/clients/listener"
	"code.justin.tv/devhub/e2ml/libs/discovery/broker/deferred"
	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/metrics/devnull"
	"code.justin.tv/devhub/e2ml/libs/setup"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/auth/fake"
	"code.justin.tv/devhub/e2ml/libs/stream/registry"
	"code.justin.tv/devhub/e2ml/libs/timeout"
	"code.justin.tv/devhub/e2ml/libs/websocket"
	"github.com/alexflint/go-arg"
)

type Config struct {
	ServerURL setup.URL     `arg:"--server" default:"wss://localhost:3002" help:"server url for connection (https for discovered, wss for direct)"`
	Log       logging.Level `default:"info" help:"set logging level [trace,debug,info,warning,error]"`
	Insecure  bool          `help:"FOR LOCAL ONLY: allow untrusted server connections"`
	Address   string        `default:"loadtest@1?n=0" help:"address for listener to consume"`
	Verbose   bool          `help:"display full message content"`
}

func main() {
	var config Config
	arg.MustParse(&config)

	logger := setup.Logger(config.Log)

	addr, err := stream.ParseAddress(config.Address)
	if err != nil {
		setup.PanicWithMessage("Illegal address requested", err)
	}

	mgr := lifecycle.NewManager()
	mgr.SetOnError(func(x interface{}, err error) { logger(logging.Error, "Shutdown error for %#v: %v", x, err) })
	mgr.SetOnPanic(func(x interface{}, err error) { logger(logging.Error, "Panic: %v", err) })
	defer func() { _ = mgr.ExecuteAll() }()

	logger(logging.Info, "Connecting to", config.ServerURL, "to write to", addr)

	resolver := websocket.NewClientResolver(&websocket.Settings{
		Certs:     &tls.Config{InsecureSkipVerify: config.Insecure},
		Lifecycle: mgr,
		Logger:    logger,
		Timeout:   timeout.NewConstantSampler(20 * time.Second),
	})

	reg := registry.NewRemote(
		fake.NewAuthSource(),
		deferred.NewBroker(setup.Unwrap(config.ServerURL)),
		resolver,
		time.Second,
		devnull.NewTracker(),
		logger,
	)

	reg.Reader(addr).Join(listener.NewLogging(config.Verbose))

	logger(logging.Info, "Shutdown for signal", mgr.ListenForInterrupt())
}
