package listener

import (
	"log"
	"sync"

	"code.justin.tv/devhub/e2ml/libs/stream"
)

type logging struct {
	trackers map[stream.AddressKey]*stream.Tracker
	verbose  bool
	mutex    sync.Mutex
}

func NewLogging(verbose bool) stream.Listener {
	return &logging{trackers: make(map[stream.AddressKey]*stream.Tracker)}
}

func (l *logging) tracker(addr stream.Address) *stream.Tracker {
	key := addr.Key()
	l.mutex.Lock()
	tracker, ok := l.trackers[key]
	if !ok {
		tracker = new(stream.Tracker)
		l.trackers[key] = tracker
	}
	l.mutex.Unlock()
	return tracker
}

func (l *logging) Current(addr stream.Address) (stream.SourceID, stream.Position) {
	return l.tracker(addr).Current()
}

func (l *logging) OnDataLost(desc stream.MessageDescription) bool {
	if l.tracker(desc.Address()).Accept(desc) {
		log.Printf("%v [%X] %d:%d <lost>", desc.Address(), desc.Source(), desc.At().Start, desc.At().End)
	}
	return true
}

func (l *logging) OnDataReceived(msg stream.Message) bool {
	if l.tracker(msg.Address()).Accept(msg) {
		if l.verbose {
			log.Printf("%v [%X] %d:%d %v", msg.Address(), msg.Source(), msg.At().Start, msg.At().End, msg.Data())
		} else {
			log.Printf("%v [%X] %d:%d %v", msg.Address(), msg.Source(), msg.At().Start, msg.At().End, len(msg.Data()))
		}
	}
	return true
}

func (l *logging) OnStreamClosed(addr stream.Address, err error) bool {
	l.mutex.Lock()
	delete(l.trackers, addr.Key())
	l.mutex.Unlock()
	return true
}
