package balanced

import (
	"sync"
	"time"

	"code.justin.tv/devhub/e2ml/libs/discovery"
	"code.justin.tv/devhub/e2ml/libs/discovery/broker"
	"code.justin.tv/devhub/e2ml/libs/discovery/broker/balanced/pick"
	"code.justin.tv/devhub/e2ml/libs/discovery/election"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol/message"
	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/metrics"
	"code.justin.tv/devhub/e2ml/libs/peering"
	"code.justin.tv/devhub/e2ml/libs/session"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/auth"
)

type OpaqueBytes = stream.OpaqueBytes

var unavailableScopes = broker.NewStaticScopesPromise(nil, protocol.ErrNoHostAvailable)
var unavailableTicket = broker.NewStaticTicketPromise(nil, protocol.ErrNoHostAvailable)

type stats struct {
	tracker           metrics.Tracker
	scopes            metrics.Aggregator
	electionStarts    metrics.Count
	electionSuccesses metrics.Count
	electionFailures  metrics.Count
	pickedExisting    metrics.Count
	pickedNew         metrics.Count
	pickedElection    metrics.Count
	pickUnavailable   metrics.Count
	authReserveOk     metrics.Count
	authValidateOk    metrics.Count
	authReserveFail   metrics.Count
	authValidateFail  metrics.Count
}

type brokerImpl struct {
	stats            stats
	peers            peering.Manager
	hostAuthorizer   *auth.Reader
	clientAuthorizer *auth.Reader
	peerAuthorizer   *auth.Reader
	pickRefresh      time.Duration
	bindings         *bindingManager
	elections        *electionManager
	channels         map[stream.AddressKey]*channel
	mutex            sync.RWMutex
	logger           logging.Function
}

var _ session.Server = (*brokerImpl)(nil)
var _ election.DataSource = (*brokerImpl)(nil)

func NewBroker(
	peerSrc peering.ManagerFactory,
	peerAuth stream.AuthSource,
	peerAuthorizer *auth.Reader,
	hostAuthorizer *auth.Reader,
	clientAuthorizer *auth.Reader,
	pickRefresh time.Duration,
	electionTimeout time.Duration,
	tracker metrics.Tracker,
	logger logging.Function,
) (discovery.BrokerServer, error) {
	b := &brokerImpl{
		stats: stats{
			tracker:           tracker,
			scopes:            tracker.Aggregator("broker.scopes", []string{}),
			electionStarts:    tracker.Count("broker.elections", []string{"action:start"}),
			electionSuccesses: tracker.Count("broker.elections", []string{"action:complete"}),
			electionFailures:  tracker.Count("broker.elections", []string{"action:fail"}),
			authReserveOk:     tracker.Count("broker.auth", []string{"action:reserve", "result:ok"}),
			authReserveFail:   tracker.Count("broker.auth", []string{"action:reserve", "result:fail"}),
			authValidateOk:    tracker.Count("broker.auth", []string{"action:validate", "result:ok"}),
			authValidateFail:  tracker.Count("broker.auth", []string{"action:validate", "result:fail"}),
			pickedExisting:    tracker.Count("broker.pick", []string{"result:existing"}),
			pickedNew:         tracker.Count("broker.pick", []string{"result:new"}),
			pickedElection:    tracker.Count("broker.pick", []string{"result:election"}),
			pickUnavailable:   tracker.Count("broker.pick", []string{"result:unavailable"}),
		},
		hostAuthorizer:   hostAuthorizer,
		clientAuthorizer: clientAuthorizer,
		peerAuthorizer:   peerAuthorizer,
		pickRefresh:      pickRefresh,
		channels:         make(map[stream.AddressKey]*channel),
		logger:           logger,
	}

	b.bindings = newBindingManager(tracker, b.onNewReporter, b.onNewPeer, peerAuth)
	b.peers = peerSrc(b.bindings.createPeer)
	b.elections = newElectionManager(b.peers.LocalName(), b, b.sendAllForElection, electionTimeout, logger)
	if err := b.peers.Start(); err != nil {
		return nil, err
	}
	return b, nil
}

func (b *brokerImpl) Close() error {
	return b.elections.shutdown()
}

func (b *brokerImpl) FindHost(auth stream.AuthRequest, addr stream.Address) (discovery.Ticket, error) {
	creds, err := b.resolve(auth)
	if err != nil {
		return nil, err
	}
	return b.findHostWithResolvedCreds(creds, addr).Result()
}

func (b *brokerImpl) findHostWithResolvedCreds(creds stream.Credentials, addr stream.Address) discovery.TicketPromise {
	host, matched := b.bestHost(addr) // consider caching
	if host == nil {
		b.stats.pickUnavailable.Add(1)
		return unavailableTicket
	}
	if matched {
		b.stats.pickedExisting.Add(1)
		return host.Reserve(creds, addr)
	}
	if !host.Status().Flags().IsSource() {
		b.stats.pickedNew.Add(1)
		return host.Reserve(creds, addr)
	}
	b.stats.pickedElection.Add(1)

	promise, created := b.elections.start(addr)
	if created {
		b.stats.electionStarts.Add(1)
	}
	return promise.Reserve(creds, addr)
}

// channel returns an existing channel or creates a new one
func (b *brokerImpl) channel(scope stream.AddressScope) (*channel, bool) {
	addrKey := scope.Key()
	ch, ok := b.channels[addrKey]
	if ok {
		return ch, false // cached
	}

	ch = newChannel(scope, stream.AncestorList(scope), b.pickRefresh, b.stats.tracker, b.logger)
	b.channels[addrKey] = ch
	return ch, true // new channel created
}

func (b *brokerImpl) addHosts(host pick.Host, scopes stream.AddressSourceMap) {
	newScopesAdded := int64(0)
	collisions := map[stream.AddressKey][]pick.Entry{}

	b.mutex.Lock()
	for addrKey, sourceID := range scopes {
		scope, err := addrKey.ToAddressScope()
		if err != nil {
			b.logger(logging.Warning, "Illegal scope added by host: ", addrKey)
			continue
		}
		ch, isNew := b.channel(scope)
		if isNew {
			newScopesAdded++
		}

		// Set election promise as successful. At this point, the Source is ready to serve this address
		b.elections.onSuccess(addrKey, host)

		// Check for collisions
		// Each channel should have only one host/sourceID; if there are more, they are collisions.
		entryHosts, _ := ch.add(host, sourceID)
		if len(entryHosts) > 1 {
			// Keep the host entry with smallest sourceID, and mark the others for collision auto-fix.
			// NOTE: ch.add method returns the new list already sorted by sourceID.
			// This is important to ensure the same sourceID is elected on all Pathfinder peer instances.
			collisions[addrKey] = entryHosts[1:]
		}
	}
	b.mutex.Unlock()

	b.stats.scopes.Add(newScopesAdded)

	// Collisions auto-fix: Send <detach> to the hosts that are hanling the same address so they stop doing it.
	for addrKey, entryHosts := range collisions {
		if addr, err := addrKey.ToAddress(); err == nil {
			for _, entry := range entryHosts {
				entry.Host().Detach(addr)
			}
		}
	}
}

func (b *brokerImpl) removeHosts(host pick.Host, scopes stream.AddressSourceMap) {
	deltaScopes := int64(0)
	b.mutex.Lock()
	for addrKey := range scopes {
		ch, ok := b.channels[addrKey]
		if ok {
			if _, empty := ch.remove(host); empty {
				delete(b.channels, addrKey)
				deltaScopes--
			}
		}
	}
	b.mutex.Unlock()
	b.stats.scopes.Add(deltaScopes)
}

func (b *brokerImpl) handleHostClosed(host pick.Host, scopes stream.AddressSourceMap) {
	b.removeHosts(host, scopes)
	scopeMsg, err := message.NewScopes(protocol.FirstAckID, scopes, true)
	if err != nil {
		b.logger(logging.Warning, "Unable to report host closure to peers: failed to create message: ", err)
		return
	}
	b.forwardAll(host, scopeMsg, message.StatusClosed)
}

func (b *brokerImpl) forwardAll(host pick.Host, msgs ...protocol.Message) {
	for _, msg := range msgs {
		if msg, err := message.NewForward(false, host.RemoteAddress(), msg); err == nil {
			if bytes, err := msg.Marshal(protocol.BroadcastSafe); err == nil {
				b.peers.BroadcastBinary(bytes)
			}
		}
	}
}

func (b *brokerImpl) sendAllForElection(msgs ...protocol.Message) {
	for _, msg := range msgs {
		if bytes, err := msg.Marshal(protocol.BroadcastSafe); err == nil {
			b.peers.BroadcastBinary(bytes)
			b.elections.WriteBinary(bytes)
		}
	}
}

func (b *brokerImpl) authHost(code OpaqueBytes) error {
	_, err := b.hostAuthorizer.Read(code)
	return err
}

func (b *brokerImpl) authPeer(code OpaqueBytes) error {
	_, err := b.peerAuthorizer.Read(code)
	return err
}

func (b *brokerImpl) resolve(req stream.AuthRequest) (stream.Credentials, error) {
	creds, err := b.clientAuthorizer.Resolve(req)
	if err != nil {
		b.stats.authReserveFail.Add(1)
	} else {
		b.stats.authReserveOk.Add(1)
	}
	return creds, err
}

func (b *brokerImpl) validate(code OpaqueBytes) (stream.Credentials, error) {
	creds, err := b.clientAuthorizer.Read(code)
	if err != nil {
		b.stats.authValidateFail.Add(1)
	} else {
		b.stats.authValidateOk.Add(1)
	}
	return creds, err
}

func (b *brokerImpl) Factory() session.BindingFactory {
	return func(client session.Client) session.Binding { return b.bindings.createReporter(client) }
}

func (b *brokerImpl) onNewReporter(client session.Client) *reporterBinding {
	return newReporterBinding(b.addHosts, b.removeHosts, b.handleHostClosed, b.forwardAll, b.bindings.removeReporter, b.authHost, b.validate, b.logger, client)
}

func (b *brokerImpl) onNewPeer(client session.Client) *peerBinding {
	return newPeerBinding(b.addHosts, b.removeHosts, b.elections.execute, b.bindings.removePeer, b.authPeer, b.bindings.initializePeer, b.bindings.findReporter, b.logger, client)
}

func (b *brokerImpl) bestHost(addr stream.Address) (pick.Host, bool) {
	var host pick.Host
	matched := false
	b.mutex.RLock()
	defer b.mutex.RUnlock()

	if ch, ok := b.channels[addr.Key()]; ok {
		if host, ok = ch.pick(); ok {
			matched = true
		}
	}
	if !matched { // look on ancestor scopes (removing filters)
		for _, a := range stream.AncestorList(addr) {
			if ch, ok := b.channels[a.Key()]; ok {
				if host, ok = ch.pick(); ok {
					break
				}
			}
		}
	}
	return host, matched
}

// used on election accepted
func (b *brokerImpl) LocalChoice(addr stream.Address) (string, bool) {
	if host, _ := b.bestHost(addr); host != nil {
		return host.Hostname(), true
	}
	return "", false
}

func (b *brokerImpl) OnChosen(addr stream.Address, hostname string) {
	b.elections.onClose.RunUntilComplete(func() {
		addrKey := addr.Key()
		host, found := b.findHost(addr, hostname)
		if !found {
			b.elections.onFailure(addrKey, protocol.ErrNoHostAvailable)
			b.stats.electionFailures.Add(1)
			return
		}

		_, err := host.Allocate(addr).Result()
		if err != nil {
			b.elections.onFailure(addrKey, err)
			b.stats.electionFailures.Add(1)
			return
		}

		b.stats.electionSuccesses.Add(1)
		// Note: b.elections.onSuccess(addrKey, host) is called later on addHost
	})
}

func (b *brokerImpl) findHost(addr stream.Address, hostname string) (host pick.Host, found bool) {
	b.mutex.RLock()
	defer b.mutex.RUnlock()
	ch, ok := b.channels[addr.Key()]
	if ok {
		host, found = ch.findSource(hostname)
	}
	if !found {
		for _, a := range stream.AncestorList(addr) {
			if ch, ok = b.channels[a.Key()]; ok {
				if host, found = ch.findSource(hostname); found {
					break
				}
			}
		}
	}
	return
}

func (b *brokerImpl) QuorumSize() int { return b.peers.DiscoveredCount()/2 + 1 }
