package balanced

import (
	"sync/atomic"
	"time"
	"unsafe"

	"code.justin.tv/devhub/e2ml/libs/discovery/broker/balanced/pick"
	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/metrics"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

type updateFunc func(pick.List) (pick.List, pick.AddFlags, bool)

type cstats struct {
	options    metrics.Aggregator
	collisions metrics.Aggregator
}

type channel struct {
	stats     cstats
	scope     stream.AddressScope
	ancestors stream.AddressScopes
	picks     unsafe.Pointer // pick.List
	logger    logging.Function
	collision int32
}

func newChannel(scope stream.AddressScope, ancestors stream.AddressScopes, period time.Duration, stats metrics.Tracker, logger logging.Function) *channel {
	list := pick.NewList(period)
	return &channel{
		stats: cstats{
			options:    stats.Aggregator("broker.options", []string{}),
			collisions: stats.Aggregator("broker.collisions", []string{}),
		},
		scope:     scope,
		ancestors: ancestors,
		picks:     unsafe.Pointer(&list),
		logger:    logger,
	}
}

func (c *channel) pick() (pick.Host, bool) {
	list, _, _ := c.updateOrRetry(func(l pick.List) (pick.List, pick.AddFlags, bool) {
		list, _ := l.Tick()
		return list, pick.None, false
	})
	return list.Pick()
}

func (c *channel) add(host pick.Host, source stream.SourceID) ([]pick.Entry, bool) {
	list, flags, updated := c.updateOrRetry(addHost(host, source))
	if flags == pick.Appended {
		c.stats.options.Add(1)
		if _, ok := c.scope.(stream.Address); ok && list.HasCollision() && atomic.SwapInt32(&c.collision, 1) == 0 {
			c.stats.collisions.Add(1)
			c.logger(logging.Warning, "Collision detected", c.scope, list)
		}
	}
	if updated {
		c.logger(logging.Trace, "Brk [ADD]", c.scope, host, list)
	}
	sortedPicks := list.GetSourceEntries() // filter available sources, sorted by SourceID
	return sortedPicks, flags == pick.Appended
}

func addHost(h pick.Host, s stream.SourceID) updateFunc {
	return func(l pick.List) (pick.List, pick.AddFlags, bool) { return l.Add(h, s) }
}

func (c *channel) remove(host pick.Host) (bool, bool) {
	list, _, updated := c.updateOrRetry(removeHost(host))
	if updated {
		c.stats.options.Add(-1)
		if _, ok := c.scope.(stream.Address); ok && !list.HasCollision() && atomic.SwapInt32(&c.collision, 0) == 1 {
			c.stats.collisions.Add(-1)
			c.logger(logging.Warning, "Collision resolved", c.scope, list)
		}
		c.logger(logging.Trace, "Brk [DROP]", c.scope, host, list)
	}
	return updated, list.IsEmpty()
}

func removeHost(h pick.Host) updateFunc {
	return func(l pick.List) (pick.List, pick.AddFlags, bool) {
		list, changed := l.Remove(h)
		return list, pick.None, changed
	}
}

func (c *channel) findSource(hostname string) (pick.Host, bool) {
	content := atomic.LoadPointer(&c.picks)
	list := *(*pick.List)(content)
	return list.Find(hostname)
}

func (c *channel) updateOrRetry(updater updateFunc) (pick.List, pick.AddFlags, bool) {
	for {
		prev := atomic.LoadPointer(&c.picks)
		cast := *(*pick.List)(prev)
		current, flags, updated := updater(cast)
		if !updated || atomic.CompareAndSwapPointer(&c.picks, prev, unsafe.Pointer(&current)) {
			return current, flags, updated
		}
	}
}
