package balanced

import (
	"net"
	"sync"
	"time"

	"code.justin.tv/devhub/e2ml/libs/discovery/broker/balanced/pick"
	"code.justin.tv/devhub/e2ml/libs/discovery/election"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol/message"
	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/session"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/lib-lifecycle/src/lifecycle"
)

type broadcastHandler func(...protocol.Message)

type electionManager struct {
	onClose   lifecycle.Manager
	proposer  election.Proposer
	acceptor  election.Acceptor
	learner   election.Learner
	sendAll   broadcastHandler
	logger    logging.Function
	elections *hostPromiseMap
	mutex     sync.Mutex
}

func newElectionManager(
	name string,
	data election.DataSource,
	sendAll func(...protocol.Message),
	timeout time.Duration,
	logger logging.Function,
) *electionManager {
	e := &electionManager{
		onClose:   lifecycle.NewManager(),
		proposer:  election.NewProposer(name, data),
		acceptor:  election.NewAcceptor(),
		learner:   election.NewLearner(data),
		sendAll:   sendAll,
		logger:    logger,
		elections: newHostPromiseMap(),
	}
	e.onClose.TickUntilClosed(e.tick, timeout)
	return e
}

func (e *electionManager) tick() {
	e.proposer.Expire()
	e.acceptor.Expire()
	e.learner.Expire()
	e.mutex.Lock()
	expired := e.elections.tickExpire()
	e.mutex.Unlock()
	for _, p := range expired {
		p.Finalize() // mark failed if we haven't gotten success yet
	}
}

func (e *electionManager) start(addr stream.Address) (*asyncHostPromise, bool) {
	e.mutex.Lock()
	promise, created := e.elections.findOrAdd(addr.Key())
	e.mutex.Unlock()
	if created {
		e.broadcast(e.proposer.Prepare(addr))
	}
	return promise, created
}

func (e *electionManager) shutdown() error {
	return e.onClose.ExecuteAll()
}

func (e *electionManager) onSuccess(addrKey stream.AddressKey, host pick.Host) {
	e.mutex.Lock()
	promise, ok := e.elections.find(addrKey)
	e.mutex.Unlock()
	if ok {
		promise.Set(host)
	}
}

func (e *electionManager) onFailure(addrKey stream.AddressKey, err error) {
	e.mutex.Lock()
	promise, ok := e.elections.find(addrKey)
	e.mutex.Unlock()
	if ok {
		promise.OnError(err)
	}
}

func (e *electionManager) execute(version protocol.Version, src session.Client, msg protocol.Message) {
	switch msg.OpCode() {
	case protocol.Prepare:
		if prep, ok := msg.(message.Prepare); ok {
			if prom, err, ok := e.acceptor.OnPrepare(prep); ok {
				e.send(version, src, prom, err)
			}
			return
		}
	case protocol.Promise:
		if prom, ok := msg.(message.Promise); ok {
			if acc, err, ok := e.proposer.OnPromise(prom); ok && err == nil {
				e.broadcast(acc, err)
			}
			return
		}
	case protocol.Accept:
		if acc, ok := msg.(message.Accept); ok {
			if accd, err, ok := e.acceptor.OnAccept(acc); ok {
				e.broadcast(accd, err)
			}
			return
		}
	case protocol.Accepted: // paxos 4
		if acc, ok := msg.(message.Accepted); ok {
			e.learner.OnAccepted(acc)
			return
		}
	}
}

func (e *electionManager) broadcast(msg protocol.Message, err error) {
	if err != nil {
		e.logger(logging.Debug, "Unable to broadcast for election: ", err)
		return
	}
	e.sendAll(msg)
}

func (e *electionManager) send(version protocol.Version, client session.Client, msg protocol.Message, err error) {
	var bytes []byte
	if err == nil {
		bytes, err = msg.Marshal(version)
	}
	if err == nil {
		err = client.WriteBinary(bytes)
	}
	if err != nil {
		e.logger(logging.Debug, "Unable to send election message: ", err)
	}
}

// extend session.Client and net.Addr to allow for loopback messaging
func (e *electionManager) Network() string   { return "dummy" }
func (e *electionManager) String() string    { return "local election" }
func (e *electionManager) Close(error)       {}
func (e *electionManager) Address() net.Addr { return e }
func (e *electionManager) WriteBinary(bytes []byte) error {
	msg, err := message.Unmarshal(bytes)
	if err == nil {
		e.execute(protocol.Current, e, msg)
	}
	return err
}
func (e *electionManager) WriteBinaryAsText(msg []byte) error {
	return protocol.ErrServiceUnavailable
}
func (e *electionManager) WriteText(msg string) error {
	return protocol.ErrServiceUnavailable
}
