package balanced

import (
	"net"
	"sync"
	"sync/atomic"
	"unsafe"

	"code.justin.tv/devhub/e2ml/libs/discovery"
	"code.justin.tv/devhub/e2ml/libs/discovery/broker"
	"code.justin.tv/devhub/e2ml/libs/discovery/broker/balanced/pick"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol/message"
	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

type scopeHandler func(pick.Host, stream.AddressSourceMap)
type allocateHandler func(pick.Host, stream.Address, broker.ScopesCallback)
type detachHandler func(pick.Host, stream.Address, broker.ScopesCallback)
type reserveHandler func(pick.Host, stream.Credentials, stream.Address, broker.TicketCallback)

type hostFunctions struct {
	add      scopeHandler
	remove   scopeHandler
	onClosed scopeHandler // only called in reaction to closed events
	allocate allocateHandler
	detach   detachHandler
	reserve  reserveHandler
	log      logging.Function
}

type host struct {
	remote net.Addr
	header message.AuthHost
	status unsafe.Pointer //*message.Status
	on     *hostFunctions
	scopes stream.AddressSourceMap
	mutex  sync.RWMutex
}

var _ pick.Host = (*host)(nil)

func newHost(header message.AuthHost, remote net.Addr, on *hostFunctions) *host {
	return &host{
		remote: remote,
		header: header,
		on:     on,
		scopes: make(stream.AddressSourceMap),
		status: unsafe.Pointer(&message.StatusUnknown),
	}
}

func (h *host) Hostname() string { return h.header.Hostname() }
func (h *host) String() string   { return h.Hostname() }

func (h *host) Summarize() ([]protocol.Message, error) {
	msg, err := message.NewScopes(protocol.FirstAckID, h.currentScopes(), false)
	if err != nil {
		return nil, err
	}
	return []protocol.Message{h.header, h.Status(), msg}, nil
}

func (h *host) Status() message.Status {
	return *(*message.Status)(atomic.LoadPointer(&h.status))
}

func (h *host) Allocate(address stream.Address) discovery.ScopesPromise {
	if !h.Status().Flags().IsAvailable() {
		return unavailableScopes
	}
	p := broker.NewAsyncScopesPromise()
	h.on.allocate(h, address, p.Set)
	return p
}

func (h *host) Detach(address stream.Address) {
	h.on.detach(h, address, func(scopes stream.AddressScopes, err error) {
		if err != nil {
			h.on.log(logging.Info, "Unable to detach collision: ", h.RemoteAddress(), scopes)
		}
	})
}

func (h *host) Reserve(creds stream.Credentials, address stream.Address) discovery.TicketPromise {
	if !h.Status().Flags().IsAvailable() {
		return unavailableTicket
	}
	promise := broker.NewAsyncTicketPromise()
	h.on.reserve(h, creds, address, promise.Set)
	return promise
}

func (h *host) RemoteAddress() net.Addr { return h.remote }

func (h *host) setStatus(status message.Status) {
	atomic.StorePointer(&h.status, unsafe.Pointer(&status))
}

func (h *host) currentScopes() stream.AddressSourceMap {
	h.mutex.Lock()
	scopes := make(stream.AddressSourceMap)
	for k, v := range h.scopes {
		scopes[k] = v
	}
	h.mutex.Unlock()
	return scopes
}

func (h *host) addScopes(scopes stream.AddressSourceMap) {
	h.mutex.Lock()
	updated := make(stream.AddressSourceMap)
	for k, v := range scopes {
		if held, found := h.scopes[k]; !found || held != v {
			h.scopes[k] = v
			updated[k] = v
		}
	}
	h.mutex.Unlock()
	h.on.add(h, updated)
}

func (h *host) removeScopes(scopes stream.AddressSourceMap) {
	h.mutex.Lock()
	updated := make(stream.AddressSourceMap)
	for k, v := range scopes {
		if _, found := h.scopes[k]; found {
			delete(h.scopes, k)
			updated[k] = v
		}
	}
	h.mutex.Unlock()
	h.on.remove(h, updated)
}

func (h *host) close() {
	h.setStatus(message.StatusClosed)
	h.mutex.Lock()
	updated := h.scopes
	h.scopes = make(stream.AddressSourceMap)
	h.mutex.Unlock()
	h.on.onClosed(h, updated)
}

func (h *host) createTicket(msg message.Ticket) discovery.Ticket {
	return broker.NewTicket(h.header.Hostname(), stream.Reservation, msg.AccessCode(), msg.Scopes())
}
