package balanced

import (
	"sync/atomic"
	"unsafe"

	"code.justin.tv/devhub/e2ml/libs/discovery"
	"code.justin.tv/devhub/e2ml/libs/discovery/broker"
	"code.justin.tv/devhub/e2ml/libs/discovery/broker/balanced/pick"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/promise"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

var (
	unset = unsafe.Pointer(&protocol.ErrServiceTimedOut)
)

type asyncHostPromise struct {
	src promise.MutableAny
	err unsafe.Pointer
}

func newAsyncHostPromise() *asyncHostPromise {
	return &asyncHostPromise{promise.NewAny(), unset}
}

func (a *asyncHostPromise) WouldBlock() bool          { return a.src.WouldBlock() }
func (a *asyncHostPromise) IsDone() bool              { return !a.src.WouldBlock() }
func (a *asyncHostPromise) Get() (interface{}, error) { return a.src.Get() }
func (a *asyncHostPromise) Result() (pick.Host, error) {
	result, err := a.src.Get()
	if cast, ok := result.(pick.Host); ok || result == nil {
		return cast, err
	}
	return nil, protocol.ErrInvalidCacheEntry
}

func (a *asyncHostPromise) Set(result pick.Host) {
	a.src.Set(result, nil)
}

// Record first error if one occurs, future errors will not overwrite
func (a *asyncHostPromise) OnError(err error) {
	atomic.CompareAndSwapPointer(&a.err, unset, unsafe.Pointer(&err))
}

// On timeout, respond with first error if not already set with a success, falls back to timeout
// error if no success or error was ever set.  It's possible that local resolution fails with an
// error, but the host is later forwarded from another node successfully -- we want to accept the
// host when that happens, which is particularly likely when the cluster is rolling over.
func (a *asyncHostPromise) Finalize() {
	// only updates if not already set
	a.src.Set(nil, *(*error)(atomic.LoadPointer(&a.err)))
}

func (a *asyncHostPromise) Reserve(creds stream.Credentials, addr stream.Address) discovery.TicketPromise {
	host, err := a.Result()
	if err != nil {
		return broker.NewStaticTicketPromise(nil, err)
	}
	return host.Reserve(creds, addr)
}
