package pick

import (
	"net"
	"testing"

	"code.justin.tv/devhub/e2ml/libs/discovery"
	"code.justin.tv/devhub/e2ml/libs/discovery/broker"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol/message"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
)

type mockHost struct {
	load  protocol.LoadFactor
	flags protocol.StatusFlags
}

func (*mockHost) Hostname() string { return "mock" }

func (*mockHost) Allocate(stream.Address) discovery.ScopesPromise {
	return broker.NewStaticScopesPromise(nil, protocol.ErrServiceUnavailable)
}

func (*mockHost) Detach(stream.Address) {}

func (*mockHost) Reserve(stream.Credentials, stream.Address) discovery.TicketPromise {
	return broker.NewStaticTicketPromise(nil, protocol.ErrServiceUnavailable)
}

func (m *mockHost) Status() message.Status {
	msg, _ := message.NewStatus(m.load, m.flags)
	return msg
}

func (m *mockHost) RemoteAddress() net.Addr {
	return &net.TCPAddr{}
}

func TestEntry(t *testing.T) {
	t.Run("should clear weight if balance is called without a score", func(t *testing.T) {
		entry := newEntry(&mockHost{}, stream.None, noScore, maxScore)
		entry.balance(10, 0)
		assert.Equal(t, noScore, entry.weight)
	})

	t.Run("should set weight if balance is called with a score", func(t *testing.T) {
		entry := newEntry(&mockHost{}, stream.None, minScore, noScore)
		target := uint64(10)
		entry.balance(target, 0)
		assert.Equal(t, target-entry.score, entry.weight)
	})

	t.Run("should set weight if balance is called with a score + scale", func(t *testing.T) {
		entry := newEntry(&mockHost{}, stream.None, minScore, noScore)
		target := uint64(10)
		scale := 1
		entry.balance(target, scale)
		assert.Equal(t, (target-entry.score)>>scale, entry.weight)
	})

	t.Run("clone should copy values but leave the original untouched", func(t *testing.T) {
		host := &mockHost{}
		entry := newEntry(host, stream.None+3, minScore, minScore)
		e2 := entry.clone()
		assert.False(t, entry == e2)
		assert.Equal(t, entry, e2)
		assert.True(t, host == e2.Host())
		assert.Equal(t, stream.None+3, e2.Source())
		e2.score = maxScore
		assert.NotEqual(t, entry, e2)
	})

	t.Run("refresh should update the store to match current host status", func(t *testing.T) {
		host := &mockHost{protocol.MaxLoad, protocol.Available}
		entry := newEntry(host, stream.None, minScore, minScore)
		entry.refresh()
		assert.Equal(t, statusToScore(host.Status()), entry.score)
	})
}
