package pick

import (
	"fmt"
	"testing"
	"time"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol/message"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestList(t *testing.T) {
	t.Run("pick should return false for an empty list", func(t *testing.T) {
		list := NewList(time.Hour)
		assert.True(t, list.IsEmpty())
		assert.False(t, list.HasCollision())
		val, ok := list.Pick()
		assert.False(t, ok)
		assert.Nil(t, val)

		val, ok = list.Find("mock")
		assert.False(t, ok)
		assert.Nil(t, val)
	})

	t.Run("pick should return properly for single item lists", func(t *testing.T) {
		host := &mockHost{protocol.MaxLoad, protocol.Available}
		list, flags, updated := NewList(time.Hour).Add(host, stream.None)
		require.True(t, updated)
		assert.Equal(t, Appended, flags)
		val, ok := list.Pick()
		assert.True(t, ok)
		assert.True(t, host == val)
		assert.False(t, list.IsEmpty())
		assert.False(t, list.HasCollision())
	})

	t.Run("pick should return weighted item for multi-item lists", func(t *testing.T) {
		host := &mockHost{protocol.MaxLoad, protocol.Available}
		host2 := &mockHost{protocol.NoLoad, protocol.Available}
		list, flags, updated := NewList(time.Hour).Add(host, stream.None)
		require.True(t, updated)
		assert.Equal(t, Appended, flags)
		list, flags, updated = list.Add(host2, stream.None)
		require.True(t, updated)
		assert.Equal(t, Appended, flags)
		val, ok := list.Pick()
		assert.True(t, ok)
		assert.True(t, host2 == val)
		assert.False(t, list.IsEmpty())
		assert.False(t, list.HasCollision())
	})

	t.Run("pick should ignore draining hosts", func(t *testing.T) {
		host := &mockHost{protocol.MaxLoad, protocol.Draining}
		host2 := &mockHost{protocol.NoLoad, protocol.Draining}
		list, flags, updated := NewList(time.Hour).Add(host, stream.None)
		require.True(t, updated)
		assert.Equal(t, Appended, flags)
		list, flags, updated = list.Add(host2, stream.None)
		require.True(t, updated)
		assert.Equal(t, Appended, flags)
		val, ok := list.Pick()
		assert.False(t, ok)
		assert.Nil(t, val)
		assert.False(t, list.IsEmpty())
		assert.False(t, list.HasCollision())
	})

	t.Run("pick should panic if an impossible weight is given", func(t *testing.T) {
		host := &mockHost{protocol.NoLoad, protocol.Draining}
		out, _, updated := NewList(time.Hour).Add(host, stream.None)
		require.True(t, updated)
		cast, ok := out.(*list)
		require.True(t, ok)
		cast.weight = 1
		assert.Panics(t, func() { cast.Pick() })
	})

	t.Run("hasCollision should return true if multiple sources are present", func(t *testing.T) {
		list, flags, updated := NewList(time.Hour).Add(&mockHost{protocol.NoLoad, protocol.Available | protocol.Source}, stream.None)
		require.True(t, updated)
		assert.Equal(t, Appended, flags)
		list, flags, updated = list.Add(&mockHost{protocol.NoLoad, protocol.Available | protocol.Source}, stream.None)
		require.True(t, updated)
		assert.Equal(t, Appended, flags)
		assert.True(t, list.HasCollision())
	})

	t.Run("GetSourceEntries should return hosts marked source sorted by sourceID", func(t *testing.T) {
		list, flags, updated := NewList(time.Hour).Add(&mockHost{protocol.NoLoad, protocol.Available | protocol.Source}, stream.None+5)
		require.True(t, updated)
		assert.Equal(t, Appended, flags)
		list, flags, updated = list.Add(&mockHost{protocol.NoLoad, protocol.Available | protocol.Source}, stream.None)
		require.True(t, updated)
		assert.Equal(t, Appended, flags)
		list, flags, updated = list.Add(&mockHost{protocol.NoLoad, protocol.Available | protocol.Source}, stream.None+3)
		require.True(t, updated)
		assert.Equal(t, Appended, flags)
		entries := list.GetSourceEntries()
		require.Len(t, entries, 3)
		assert.Equal(t, stream.None, entries[0].Source())
		assert.Equal(t, stream.None+3, entries[1].Source())
		assert.Equal(t, stream.None+5, entries[2].Source())
	})

	t.Run("tick should return original list if empty", func(t *testing.T) {
		list := NewList(time.Hour)
		result, copied := list.Tick()
		assert.False(t, copied)
		assert.True(t, result == list)
	})

	t.Run("tick should return original if not expired", func(t *testing.T) {
		host := &mockHost{protocol.NoLoad, protocol.Draining}
		list, _, _ := NewList(time.Hour).Add(host, stream.None)
		result, copied := list.Tick()
		assert.False(t, copied)
		assert.True(t, result == list)
	})

	t.Run("tick should clone if expired", func(t *testing.T) {
		host := &mockHost{protocol.NoLoad, protocol.Draining}
		list, _, _ := NewList(time.Nanosecond).Add(host, stream.None)
		time.Sleep(time.Millisecond)
		result, copied := list.Tick()
		assert.True(t, copied)
		assert.False(t, result == list)
	})

	t.Run("find should return false if the host is missing", func(t *testing.T) {
		host := &mockHost{protocol.NoLoad, protocol.NoFlags}
		list, _, updated := NewList(time.Nanosecond).Add(host, stream.None)
		require.True(t, updated)
		val, found := list.Find(host.Hostname() + "_not_found")
		assert.Nil(t, val)
		assert.False(t, found)
	})

	t.Run("find should return the host if found", func(t *testing.T) {
		host := &mockHost{protocol.NoLoad, protocol.NoFlags}
		list, _, updated := NewList(time.Nanosecond).Add(host, stream.None)
		require.True(t, updated)
		val, found := list.Find(host.Hostname())
		assert.Equal(t, host, val)
		assert.True(t, found)
	})

	t.Run("add should do nothing if the host is present", func(t *testing.T) {
		host := &mockHost{protocol.NoLoad, protocol.Draining}
		list, flags, updated := NewList(time.Nanosecond).Add(host, stream.None)
		require.True(t, updated)
		require.Equal(t, Appended, flags)
		result, flags, updated := list.Add(host, stream.None)
		assert.True(t, result == list)
		assert.False(t, updated)
		require.Equal(t, None, flags)
	})

	t.Run("add update the host list and return no append if the host changed sources", func(t *testing.T) {
		host := &mockHost{protocol.NoLoad, protocol.Draining}
		list, flags, updated := NewList(time.Nanosecond).Add(host, stream.None)
		require.True(t, updated)
		require.Equal(t, Appended, flags)
		result, flags, updated := list.Add(host, stream.None+1)
		assert.False(t, result == list)
		assert.True(t, updated)
		require.Equal(t, None, flags)
	})

	t.Run("remove should do nothing if the host is absent", func(t *testing.T) {
		host := &mockHost{protocol.NoLoad, protocol.Draining}
		list := NewList(time.Nanosecond)
		result, updated := list.Remove(host)
		assert.True(t, result == list)
		assert.False(t, updated)
	})

	t.Run("remove should remove a host if present", func(t *testing.T) {
		host := &mockHost{protocol.NoLoad, protocol.Available}
		list, _, updated := NewList(time.Nanosecond).Add(host, stream.None)
		require.True(t, updated)
		result, updated := list.Remove(host)
		assert.False(t, result == list)
		assert.True(t, updated)
		_, ok := result.Pick()
		assert.False(t, ok)
	})

	t.Run("should list host names in debug", func(t *testing.T) {
		host := &mockHost{protocol.NoLoad, protocol.Available}
		list, _, updated := NewList(time.Nanosecond).Add(host, stream.None)
		require.True(t, updated)
		assert.Equal(t, "[mock]", fmt.Sprintf("%v", list))
	})
}

func TestStatusToScore(t *testing.T) {
	msg, _ := message.NewStatus(protocol.LoadFactor(10), protocol.Available)
	assert.Equal(t, uint64(11), statusToScore(msg))
	msg, _ = message.NewStatus(protocol.NoLoad, protocol.Available)
	assert.Equal(t, minScore, statusToScore(msg))
	msg, _ = message.NewStatus(protocol.NoLoad, protocol.Draining)
	assert.Equal(t, noScore, statusToScore(msg))
}
