package balanced

import (
	"sync"
	"time"

	"code.justin.tv/devhub/lib-lifecycle/src/lifecycle"
	"code.justin.tv/devhub/e2ml/libs/discovery"
	"code.justin.tv/devhub/e2ml/libs/discovery/broker"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

type requestManager struct {
	onClose       lifecycle.Manager
	allocs        *allocMap
	requests      *requestMap
	nextRequestID protocol.RequestID
	mutex         sync.Mutex
}

func newRequestManager(timeout time.Duration) *requestManager {
	mgr := &requestManager{
		onClose:       lifecycle.NewManager(),
		allocs:        newAllocMap(),
		requests:      newRequestMap(),
		nextRequestID: protocol.FirstRequestID,
	}
	mgr.onClose.TickUntilClosed(mgr.tick, timeout)
	return mgr
}

func (r *requestManager) tick() {
	r.mutex.Lock()
	requests := r.requests.tick()
	allocs := r.allocs.tick()
	r.mutex.Unlock()
	for _, req := range requests {
		req(nil, protocol.ErrServiceTimedOut)
	}
	for _, req := range allocs {
		req(nil, protocol.ErrServiceTimedOut)
	}
}

func (r *requestManager) close() {
	r.tick()
	r.tick()
	_ = r.onClose.ExecuteAll()
}

func (r *requestManager) bindAlloc(handler broker.ScopesCallback) protocol.RequestID {
	r.mutex.Lock()
	id := r.nextRequestID
	r.nextRequestID = id.Next()
	r.allocs.add(id, handler)
	r.mutex.Unlock()
	return id
}

func (r *requestManager) bind(handler broker.TicketCallback) protocol.RequestID {
	r.mutex.Lock()
	id := r.nextRequestID
	r.nextRequestID = id.Next()
	r.requests.add(id, handler)
	r.mutex.Unlock()
	return id
}

func (r *requestManager) onScopesSuccess(id protocol.RequestID, scopes stream.AddressScopes) {
	r.mutex.Lock()
	req, ok := r.allocs.remove(id)
	r.mutex.Unlock()
	if ok {
		req(scopes, nil)
	}
}

func (r *requestManager) onTicketSuccess(id protocol.RequestID, ticket discovery.Ticket) {
	r.mutex.Lock()
	req, ok := r.requests.remove(id)
	r.mutex.Unlock()
	if ok {
		req(ticket, nil)
	}
}

func (r *requestManager) onFailure(id protocol.RequestID, err error) {
	r.mutex.Lock()
	request, _ := r.requests.remove(id)
	scopes, _ := r.allocs.remove(id)
	r.mutex.Unlock()
	if request != nil {
		request(nil, err)
	}
	if scopes != nil {
		scopes(nil, err)
	}
}
