package http

import (
	"encoding/json"
	"io"
	"io/ioutil"
	"net/http"
	"net/url"

	"code.justin.tv/devhub/e2ml/libs/discovery"
	"code.justin.tv/devhub/e2ml/libs/discovery/broker"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	httpx "code.justin.tv/devhub/e2ml/libs/http"
	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

const maxBodyLength = 1024 * 10

type brokerImpl struct {
	client http.Client
	url    *url.URL
	logger logging.Function
}

func NewBroker(url *url.URL, cli http.Client, logger logging.Function) discovery.Broker {
	return &brokerImpl{cli, url, logger}
}

func (b *brokerImpl) FindHost(auth stream.AuthRequest, addr stream.Address) (discovery.Ticket, error) {
	req, err := NewGetHostRequest(b.url, addr)
	if err == nil && auth != nil {
		req.Header, err = auth.Inject(req.Header)
	}
	if err != nil {
		return nil, err
	}

	out := &getHostResponse{}
	if err = b.execute(req, "get_host", out); err != nil {
		return nil, err
	}
	scopes := make(stream.AddressScopes, len(out.Scopes))
	for i, key := range out.Scopes {
		scopes[i], err = stream.ParseAddressScope(key)
		if err != nil {
			return nil, err
		}
	}
	method, err := stream.ParseAuthMethod(out.Method)
	if err != nil {
		return nil, err
	}
	return broker.NewTicket(out.URL, method, out.AccessCode, scopes), nil
}

func (b *brokerImpl) execute(req *http.Request, statName string, out interface{}) error {
	b.logger(logging.Trace, "Brk - Executing request", req.URL)
	resp, err := b.client.Do(req)
	if err != nil {
		return err
	}

	defer resp.Body.Close()
	if httpErr, found := httpx.ExtractErrorFromResponse(resp); found {
		io.Copy(ioutil.Discard, resp.Body)
		return httpErr
	}

	if resp.StatusCode == http.StatusNoContent {
		io.Copy(ioutil.Discard, resp.Body)
		return nil
	}

	if err = json.NewDecoder(io.LimitReader(resp.Body, maxBodyLength)).Decode(out); err != nil {
		return protocol.ErrInvalidBody(err)
	}
	return nil
}
