package http

import (
	"encoding/json"
	"net/http"
	"net/url"

	"code.justin.tv/devhub/e2ml/libs/discovery"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

type getHostResponse struct {
	URL        string             `json:"host_address"`
	Method     string             `json:"method"`
	AccessCode stream.OpaqueBytes `json:"access_code"`
	Scopes     []string           `json:"scopes"`
}

func NewGetHostResponse(ticket discovery.Ticket) interface{} {
	scopes := ticket.Scopes()
	keys := make([]string, len(scopes))
	for i, scope := range scopes {
		keys[i] = string(scope.Key())
	}
	return &getHostResponse{ticket.URL(), ticket.Method().String(), ticket.AccessCode(), keys}
}

// NOTE : we're losing log safety here so that this struct can be auto-serialized by json
func (r getHostResponse) MarshalJSON() ([]byte, error) {
	return json.Marshal(map[string]interface{}{
		"host_address": r.URL,
		"method":       r.Method,
		"access_code":  string(r.AccessCode),
		"scopes":       r.Scopes,
	})
}

func NewGetHostRequest(baseURL *url.URL, addr stream.Address) (*http.Request, error) {
	reqURL := *baseURL
	reqURL.Path = protocol.HttpCurrentVersionPath + protocol.GetHostPath

	query := make(url.Values)
	query.Set(protocol.GetHostAddressParam, string(addr.Key()))
	reqURL.RawQuery = query.Encode()

	req, err := http.NewRequest(protocol.GetHostMethod, reqURL.String(), nil)
	if err != nil {
		return nil, err
	}

	return req, err
}
