package election

import (
	"sync"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol/message"
)

type Acceptor interface {
	Expirable
	OnPrepare(message.Prepare) (message.Promise, error, bool)
	OnAccept(message.Accept) (message.Accepted, error, bool)
}

type acceptor struct {
	suggestions *suggestionMap
	accepted    *acceptMap
	mutex       sync.Mutex
}

func NewAcceptor() Acceptor {
	return &acceptor{suggestions: newSuggestionMap(), accepted: newAcceptMap()}
}

func (a *acceptor) Expire() error {
	a.mutex.Lock()
	err := a.suggestions.Expire()
	if err == nil {
		err = a.accepted.Expire()
	}
	a.mutex.Unlock()
	return err
}

func (a *acceptor) OnPrepare(msg message.Prepare) (message.Promise, error, bool) {
	a.mutex.Lock()
	if !a.suggestions.set(msg.Address(), msg.ID()) {
		a.mutex.Unlock()
		return nil, nil, false
	}
	addr := msg.Address()
	proposal, _ := a.accepted.find(addr.Key())
	a.mutex.Unlock()
	out, err := message.NewPromise(addr, msg.ID(), proposal)
	return out, err, true
}

func (a *acceptor) OnAccept(msg message.Accept) (message.Accepted, error, bool) {
	addr := msg.Address()
	a.mutex.Lock()
	if !a.suggestions.allow(addr, msg.ID()) {
		a.mutex.Unlock()
		return nil, nil, false
	}
	a.accepted.set(addr.Key(), msg)
	a.mutex.Unlock()
	out, err := message.NewAccepted(addr, msg.ID(), msg.Hostname())
	return out, err, true
}
