package election

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol/message"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestAcceptor(t *testing.T) {
	addr, err := stream.NewAddress(stream.Namespace("n"), 1, map[string]string{})
	require.NoError(t, err)

	t.Run("allows monotonic increasing prepare statement ids", func(t *testing.T) {
		prepare, err := message.NewPrepare(addr, protocol.FirstSuggestion("x"))
		require.NoError(t, err)
		l := NewAcceptor()

		promise, err, ok := l.OnPrepare(prepare)
		assert.True(t, ok)
		assert.NoError(t, err)
		require.NotNil(t, promise)
		assert.Equal(t, promise.ID(), prepare.ID())

		// equal value
		promise, err, ok = l.OnPrepare(prepare)
		assert.True(t, ok)
		assert.NoError(t, err)
		require.NotNil(t, promise)
		assert.Equal(t, promise.ID(), prepare.ID())

		// greater value
		prepare2, _ := message.NewPrepare(addr, prepare.ID().Next())
		promise, err, ok = l.OnPrepare(prepare2)
		assert.True(t, ok)
		assert.NoError(t, err)
		require.NotNil(t, promise)
		assert.Equal(t, promise.ID(), prepare2.ID())

		// lesser value (blocked)
		promise, err, ok = l.OnPrepare(prepare)
		assert.False(t, ok)
		assert.NoError(t, err)
		assert.Nil(t, promise)
	})

	t.Run("allows accept attempts by default", func(t *testing.T) {
		accept, err := message.NewAccept(addr, protocol.FirstSuggestion("x"), "choice")
		require.NoError(t, err)

		l := NewAcceptor()
		accepted, err, ok := l.OnAccept(accept)
		assert.True(t, ok)
		assert.NoError(t, err)
		require.NotNil(t, accept)
		assert.Equal(t, accept.ID(), accepted.ID())
	})

	t.Run("rejects lower accept attempts", func(t *testing.T) {
		low := protocol.FirstSuggestion("x")
		high := low.Next()
		accept, err := message.NewAccept(addr, low, "choice")
		require.NoError(t, err)
		prepare, err := message.NewPrepare(addr, high)
		require.NoError(t, err)
		l := NewAcceptor()

		l.OnPrepare(prepare)
		accepted, err, ok := l.OnAccept(accept)
		assert.False(t, ok)
		assert.NoError(t, err)
		assert.Nil(t, accepted)
	})

	t.Run("expires attempts after prolonged inactivity", func(t *testing.T) {
		low := protocol.FirstSuggestion("x")
		high := low.Next()
		accept, err := message.NewAccept(addr, low, "choice")
		require.NoError(t, err)
		prepare, err := message.NewPrepare(addr, high)
		require.NoError(t, err)
		l := NewAcceptor()

		l.OnPrepare(prepare)

		l.Expire()

		accepted, err, ok := l.OnAccept(accept)
		assert.False(t, ok)
		assert.NoError(t, err)
		assert.Nil(t, accepted)

		l.Expire()

		accepted, err, ok = l.OnAccept(accept)
		assert.False(t, ok)
		assert.NoError(t, err)
		assert.Nil(t, accepted)

		l.Expire()
		l.Expire()

		accepted, err, ok = l.OnAccept(accept)
		assert.True(t, ok)
		assert.NoError(t, err)
		assert.NotNil(t, accepted)
	})
}
