package election

import (
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol/message"
)

// Choice map is used by Learners; it waits for counted accepted > quorum on a
// single suggestion id.

type choiceEntry struct {
	choice string
	count  int
}

type choiceData map[protocol.SuggestionKey]*choiceEntry

type choiceMap struct {
	source  DataSource
	recent  choiceData
	expired choiceData
}

var _ Expirable = (*choiceMap)(nil)

func newChoiceMap(source DataSource) *choiceMap {
	return &choiceMap{
		source:  source,
		recent:  make(choiceData),
		expired: make(choiceData),
	}
}

func (c *choiceMap) add(msg message.Accepted) (string, bool) {
	key := msg.ID().Key()
	entry, found := c.find(key)
	if found {
		entry.count++
	} else {
		entry = &choiceEntry{msg.Hostname(), 1}
		c.recent[key] = entry
	}
	return entry.choice, c.source.QuorumSize() == entry.count
}

func (c *choiceMap) find(key protocol.SuggestionKey) (*choiceEntry, bool) {
	entry, found := c.recent[key]
	if !found {
		entry, found = c.expired[key]
		if found {
			c.recent[key] = entry // put back into recent since still active
		}
	}
	return entry, found
}

func (c *choiceMap) Expire() error {
	c.expired = c.recent
	c.recent = make(choiceData)
	return nil
}
