package election

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol/message"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestChoiceMap(t *testing.T) {
	src := &testSource{choice: "choice", quorum: 2}
	t.Run("should respond on quorum", func(t *testing.T) {
		addr, _ := stream.NewAddress(stream.Namespace("n"), 1, map[string]string{})
		resp, _ := message.NewAccepted(addr, protocol.FirstSuggestion("x"), src.choice)
		choices := newChoiceMap(src)
		_, ok := choices.add(resp)
		assert.False(t, ok)

		key := resp.ID().Key()
		entry, ok := choices.find(key)
		require.True(t, ok)
		assert.Equal(t, 1, entry.count)

		selector, ok := choices.add(resp)
		assert.True(t, ok)
		assert.Equal(t, src.choice, selector)

		entry, ok = choices.find(key)
		require.True(t, ok)
		assert.Equal(t, 2, entry.count)
	})

	t.Run("should track suggestions by id", func(t *testing.T) {
		addr, _ := stream.NewAddress(stream.Namespace("n"), 1, map[string]string{})
		resp, _ := message.NewAccepted(addr, protocol.FirstSuggestion("x"), src.choice)
		choices := newChoiceMap(src)
		_, ok := choices.add(resp)
		assert.False(t, ok)

		resp2, _ := message.NewAccepted(addr, resp.ID().Next(), src.choice)
		_, ok = choices.add(resp2)
		assert.False(t, ok)

		assert.Len(t, choices.recent, 2)
	})

	t.Run("should remember suggestions after expire", func(t *testing.T) {
		addr, _ := stream.NewAddress(stream.Namespace("n"), 1, map[string]string{})
		resp, _ := message.NewAccepted(addr, protocol.FirstSuggestion("x"), src.choice)
		choices := newChoiceMap(src)
		_, ok := choices.add(resp)
		assert.False(t, ok)

		key := resp.ID().Key()
		entry, ok := choices.find(key)
		require.True(t, ok)
		assert.Equal(t, 1, entry.count)

		selector, ok := choices.add(resp)
		assert.True(t, ok)
		assert.Equal(t, src.choice, selector)

		assert.NoError(t, choices.Expire())

		entry, ok = choices.find(key)
		require.True(t, ok)
		assert.Equal(t, 2, entry.count)

		assert.NoError(t, choices.Expire())

		entry, ok = choices.find(key)
		require.True(t, ok)
		assert.Equal(t, 2, entry.count)
	})

	t.Run("should clear suggestions after prolonged inactivity", func(t *testing.T) {
		addr, _ := stream.NewAddress(stream.Namespace("n"), 1, map[string]string{})
		resp, _ := message.NewAccepted(addr, protocol.FirstSuggestion("x"), src.choice)
		choices := newChoiceMap(src)
		_, ok := choices.add(resp)
		assert.False(t, ok)

		key := resp.ID().Key()
		entry, ok := choices.find(key)
		require.True(t, ok)
		assert.Equal(t, 1, entry.count)

		selector, ok := choices.add(resp)
		assert.True(t, ok)
		assert.Equal(t, src.choice, selector)

		assert.NoError(t, choices.Expire())
		assert.NoError(t, choices.Expire())

		entry, ok = choices.find(key)
		require.False(t, ok)
		assert.Nil(t, entry)
	})
}
