package election

import (
	"strconv"
	"sync"
	"sync/atomic"
	"testing"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol/message"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

type testSource struct {
	choice string
	quorum int
	chosen func(stream.Address, string)
}

var _ DataSource = (*testSource)(nil)

func (t *testSource) LocalChoice(stream.Address) (string, bool) { return t.choice, true }
func (t *testSource) QuorumSize() int                           { return t.quorum }
func (t *testSource) OnChosen(a stream.Address, chosen string) {
	if t.chosen != nil {
		t.chosen(a, chosen)
	}
}

// mock cluster - wiring between components to test flow
type testCluster struct {
	proposers []Proposer
	acceptors []Acceptor
	learners  []Learner
	choice    func(stream.Address) string
	chosen    func(stream.Address, string)
}

var _ DataSource = (*testCluster)(nil)

func (t *testCluster) LocalChoice(addr stream.Address) (string, bool) {
	if t.choice != nil {
		return t.choice(addr), true
	}
	return "choice", true
}

func (t *testCluster) QuorumSize() int {
	count := len(t.acceptors)
	if count < 3 {
		return count
	}
	return count / 2
}

func (t *testCluster) OnChosen(a stream.Address, chosen string) {
	if t.chosen != nil {
		t.chosen(a, chosen)
	}
}

func (t *testCluster) request(test *testing.T, index int, addr stream.Address) {
	require.True(test, index < len(t.proposers))
	prep, err := t.proposers[index].Prepare(addr)
	require.NoError(test, err)
	t.sendPrepare(t.proposers[index], prep)
}

func (t *testCluster) sendPrepare(p Proposer, prep message.Prepare) {
	for _, a := range t.acceptors {
		go func(a Acceptor) {
			if msg, err, ok := a.OnPrepare(prep); ok && err == nil {
				t.sendPromise(p, msg)
			}
		}(a)
	}
}

func (t *testCluster) sendPromise(p Proposer, prom message.Promise) {
	go func() {
		if msg, err, ok := p.OnPromise(prom); ok && err == nil {
			t.sendAccept(p, msg)
		}
	}()
}

func (t *testCluster) sendAccept(p Proposer, acc message.Accept) {
	for _, a := range t.acceptors {
		go func(a Acceptor) {
			if msg, err, ok := a.OnAccept(acc); ok && err == nil {
				t.sendAccepted(p, msg)
			}
		}(a)
	}
}

func (t *testCluster) sendAccepted(p Proposer, acc message.Accepted) {
	for _, l := range t.learners {
		go l.OnAccepted(acc)
	}
}

func createCluster(prop, acc, learn int, choice func(stream.Address) string, chosen func(stream.Address, string)) *testCluster {
	cluster := &testCluster{choice: choice, chosen: chosen}
	for i := 0; i < prop; i++ {
		name := strconv.FormatInt(int64(i), 10)
		cluster.proposers = append(cluster.proposers, NewProposer(name, cluster))
	}
	for i := 0; i < acc; i++ {
		cluster.acceptors = append(cluster.acceptors, NewAcceptor())
	}
	for i := 0; i < learn; i++ {
		cluster.learners = append(cluster.learners, NewLearner(cluster))
	}
	return cluster
}

func TestSingleNodeElection(t *testing.T) {
	var wg sync.WaitGroup
	wg.Add(1)
	choice := func(stream.Address) string { return "x" }
	chosen := func(addr stream.Address, chosen string) {
		wg.Done()
		assert.Equal(t, choice(addr), chosen)
	}
	cluster := createCluster(1, 1, 1, choice, chosen)
	addr, _ := stream.NewAddress(stream.Namespace("n"), 1, nil)
	cluster.request(t, 0, addr)
	wg.Wait()
}

func TestMultiNodeElection(t *testing.T) {
	var wg sync.WaitGroup
	value := int64(1)
	choice := func(stream.Address) string {
		return strconv.FormatInt(atomic.LoadInt64(&value), 10)
	}
	chosen := func(addr stream.Address, chosen string) {
		assert.Equal(t, "1", chosen)
		wg.Done()
	}
	cluster := createCluster(5, 5, 5, choice, chosen)
	addr, _ := stream.NewAddress(stream.Namespace("n"), 1, nil)
	wg.Add(5)
	cluster.request(t, 2, addr)
	wg.Wait()

	wg.Add(0)
	cluster.request(t, 1, addr) // should not cause feedback (lower proposal)
	wg.Wait()

	atomic.AddInt64(&value, 1)

	wg.Add(5)
	cluster.request(t, 3, addr) // should reuse value (higher proposal)
	wg.Wait()
}
