package election

import (
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol/message"
)

// Promise map is used by Proposers; it retains the highest accepted proposal
// while counting the number of responses

type promiseEntry struct {
	recv     int
	accepted protocol.Proposal
}

var _ Expirable = (*promiseMap)(nil)

type promiseData map[protocol.SuggestionKey]*promiseEntry

type promiseMap struct {
	source  DataSource
	recent  promiseData
	expired promiseData
}

func newPromiseMap(source DataSource) promiseMap {
	return promiseMap{
		source:  source,
		recent:  make(promiseData),
		expired: make(promiseData),
	}
}

func (p *promiseMap) set(msg message.Promise) (protocol.Proposal, bool) {
	key := msg.ID().Key()
	entry, found := p.find(key)
	accepted, ok := msg.Accepted()
	if found {
		entry.recv++
		if entry.accepted == nil || ok && entry.accepted.ID().Less(accepted.ID()) {
			entry.accepted = accepted
		}
	} else {
		entry = &promiseEntry{1, accepted}
		p.recent[key] = entry
	}
	return entry.accepted, p.source.QuorumSize() == entry.recv
}

func (p *promiseMap) find(key protocol.SuggestionKey) (*promiseEntry, bool) {
	entry, found := p.recent[key]
	if !found {
		entry, found = p.expired[key]
		if found {
			p.recent[key] = entry // put back into recent since still active
		}
	}
	return entry, found
}

func (p *promiseMap) Expire() error {
	p.expired = p.recent
	p.recent = make(promiseData)
	return nil
}
