package election

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol/message"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
)

func TestPromiseMap(t *testing.T) {
	src := &testSource{choice: "choice", quorum: 2}
	t.Run("should respond on quorum", func(t *testing.T) {
		addr, _ := stream.NewAddress(stream.Namespace("n"), 1, map[string]string{})
		msg, _ := message.NewPromise(addr, protocol.FirstSuggestion("x"), nil)
		p := newPromiseMap(src)
		_, ok := p.set(msg)
		assert.False(t, ok)

		out, ok := p.set(msg)
		assert.True(t, ok)
		assert.Nil(t, out)
	})

	t.Run("should track suggestions by id", func(t *testing.T) {
		addr, _ := stream.NewAddress(stream.Namespace("n"), 1, map[string]string{})
		msg, _ := message.NewPromise(addr, protocol.FirstSuggestion("x"), nil)
		p := newPromiseMap(src)
		_, ok := p.set(msg)
		assert.False(t, ok)

		msg2, _ := message.NewPromise(addr, msg.ID().Next(), nil)
		_, ok = p.set(msg2)
		assert.False(t, ok)

		assert.Len(t, p.recent, 2)
	})

	t.Run("should return the highest proposal (test high first)", func(t *testing.T) {
		addr, _ := stream.NewAddress(stream.Namespace("n"), 1, map[string]string{})
		p := newPromiseMap(src)

		low := protocol.FirstSuggestion("x")
		high := low.Next()
		acc, _ := message.NewAccept(addr, high, "value")
		msg, _ := message.NewPromise(addr, high.Next(), acc)
		_, ok := p.set(msg)
		assert.False(t, ok)

		acc2, _ := message.NewAccept(addr, low, "value2")
		msg2, _ := message.NewPromise(addr, high.Next(), acc2)
		prop, ok := p.set(msg2)

		assert.True(t, ok)
		assert.Equal(t, prop.ID(), high)
		assert.Equal(t, prop.Hostname(), acc.Hostname())
	})

	t.Run("should return the highest proposal (test low first)", func(t *testing.T) {
		addr, _ := stream.NewAddress(stream.Namespace("n"), 1, map[string]string{})
		p := newPromiseMap(src)

		low := protocol.FirstSuggestion("x")
		high := low.Next()
		acc, _ := message.NewAccept(addr, low, "value")
		msg, _ := message.NewPromise(addr, high.Next(), acc)
		_, ok := p.set(msg)
		assert.False(t, ok)

		acc2, _ := message.NewAccept(addr, high, "value2")
		msg2, _ := message.NewPromise(addr, high.Next(), acc2)
		prop, ok := p.set(msg2)

		assert.True(t, ok)
		assert.Equal(t, prop.ID(), high)
		assert.Equal(t, prop.Hostname(), acc2.Hostname())
	})

	t.Run("should remember proposals after one expiration", func(t *testing.T) {
		addr, _ := stream.NewAddress(stream.Namespace("n"), 1, map[string]string{})
		p := newPromiseMap(src)

		low := protocol.FirstSuggestion("x")
		high := low.Next()
		acc, _ := message.NewAccept(addr, high, "value")
		msg, _ := message.NewPromise(addr, high.Next(), acc)
		_, ok := p.set(msg)
		assert.False(t, ok)

		assert.NoError(t, p.Expire())

		acc2, _ := message.NewAccept(addr, low, "value2")
		msg2, _ := message.NewPromise(addr, high.Next(), acc2)
		prop, ok := p.set(msg2)

		assert.True(t, ok)
		assert.Equal(t, prop.ID(), high)
		assert.Equal(t, prop.Hostname(), acc.Hostname())
	})

	t.Run("should forget proposals after prolonged inactivity", func(t *testing.T) {
		addr, _ := stream.NewAddress(stream.Namespace("n"), 1, map[string]string{})
		p := newPromiseMap(src)

		low := protocol.FirstSuggestion("x")
		high := low.Next()
		acc, _ := message.NewAccept(addr, high, "value")
		msg, _ := message.NewPromise(addr, high.Next(), acc)
		_, ok := p.set(msg)
		assert.False(t, ok)

		assert.NoError(t, p.Expire())
		assert.NoError(t, p.Expire())

		acc2, _ := message.NewAccept(addr, low, "value2")
		msg2, _ := message.NewPromise(addr, high.Next(), acc2)
		_, ok = p.set(msg2) // send once
		assert.False(t, ok)

		prop, ok := p.set(msg2) // send twice (should be quorum)

		assert.True(t, ok)
		assert.Equal(t, prop.ID(), low)
		assert.Equal(t, prop.Hostname(), acc2.Hostname())
	})
}
