package election

import (
	"sync"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol/message"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

type Proposer interface {
	Expirable
	Prepare(stream.Address) (message.Prepare, error)
	OnPromise(message.Promise) (message.Accept, error, bool)
}

type proposer struct {
	data     DataSource
	localID  protocol.SuggestionID
	promised promiseMap
	mutex    sync.Mutex
}

func NewProposer(name string, data DataSource) Proposer {
	return &proposer{data: data, localID: protocol.FirstSuggestion(name), promised: newPromiseMap(data)}
}

func (p *proposer) Expire() error {
	p.mutex.Lock()
	err := p.promised.Expire()
	p.mutex.Unlock()
	return err
}

func (p *proposer) Prepare(value stream.Address) (message.Prepare, error) {
	p.mutex.Lock()
	id := p.localID
	p.localID = id.Next()
	p.mutex.Unlock()
	return message.NewPrepare(value, id)
}

func (p *proposer) OnPromise(msg message.Promise) (message.Accept, error, bool) {
	p.mutex.Lock()
	accepted, ok := p.promised.set(msg)
	p.mutex.Unlock()
	if ok {
		if hostname, ok := p.findHostname(msg.Address(), accepted); ok {
			out, err := message.NewAccept(msg.Address(), msg.ID(), hostname)
			return out, err, true
		}
	}
	return nil, nil, false
}

func (p *proposer) findHostname(addr stream.Address, prop protocol.Proposal) (string, bool) {
	if prop != nil {
		return prop.Hostname(), true
	}
	return p.data.LocalChoice(addr)
}
