package election

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol/message"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestProposer(t *testing.T) {
	addr, _ := stream.NewAddress(stream.Namespace("n"), 1, map[string]string{})
	src := &testSource{choice: "choice", quorum: 2}
	t.Run("generates incrementing prepare statements", func(t *testing.T) {
		name := "1"
		expected := protocol.FirstSuggestion(name)
		p := NewProposer(name, src)

		prepare, err := p.Prepare(addr)
		assert.NoError(t, err)
		require.NotNil(t, prepare)
		assert.Equal(t, prepare.ID(), expected)

		prepare, err = p.Prepare(addr)
		assert.NoError(t, err)
		require.NotNil(t, prepare)
		assert.Equal(t, prepare.ID(), expected.Next())
	})

	t.Run("properly converts proposals to accept requests with source choice", func(t *testing.T) {
		name := "1"
		p := NewProposer(name, src)
		promise, _ := message.NewPromise(addr, protocol.FirstSuggestion(name), nil)

		accept, err, ok := p.OnPromise(promise)
		assert.False(t, ok)
		assert.NoError(t, err)
		assert.Nil(t, accept)

		accept, err, ok = p.OnPromise(promise)
		assert.True(t, ok)
		assert.NoError(t, err)
		require.NotNil(t, accept)
		assert.Equal(t, src.choice, accept.Hostname())
	})

	t.Run("properly converts proposals to accept requests with overrides", func(t *testing.T) {
		name := "1"
		prev, _ := message.NewAccepted(addr, protocol.FirstSuggestion(name), "override")
		promise, _ := message.NewPromise(addr, prev.ID().Next(), prev)

		p := NewProposer(name, src)

		accept, err, ok := p.OnPromise(promise)
		assert.False(t, ok)
		assert.NoError(t, err)
		assert.Nil(t, accept)

		accept, err, ok = p.OnPromise(promise)
		assert.True(t, ok)
		assert.NoError(t, err)
		require.NotNil(t, accept)
		assert.Equal(t, "override", accept.Hostname())
	})

	t.Run("properly flushes values", func(t *testing.T) {
		name := "1"
		prev, _ := message.NewAccepted(addr, protocol.FirstSuggestion(name), "override")
		promise, _ := message.NewPromise(addr, prev.ID().Next(), prev)

		p := NewProposer(name, src)

		accept, err, ok := p.OnPromise(promise)
		assert.False(t, ok)
		assert.NoError(t, err)
		assert.Nil(t, accept)

		assert.NoError(t, p.Expire()) // call twice to completely flush values
		assert.NoError(t, p.Expire())

		promise, _ = message.NewPromise(addr, prev.ID().Next(), nil)

		accept, err, ok = p.OnPromise(promise) // we've reset the counter
		assert.False(t, ok)
		assert.NoError(t, err)
		assert.Nil(t, accept)
	})
}
