package host

import (
	"net/url"
	"sync"
	"sync/atomic"
	"unsafe"

	"code.justin.tv/devhub/e2ml/libs/discovery"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/discovery/protocol/message"
	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/metrics"
	"code.justin.tv/devhub/e2ml/libs/session"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/ticket"
)

type scopeSource func(stream.Address, stream.Credentials) (stream.AddressScopes, error)

type stats struct {
	scopes  metrics.Aggregator
	reports metrics.Count
	load    metrics.Gauge
}

type reporter struct {
	binding        *binding
	supported      stream.AddressSourceMap
	status         unsafe.Pointer
	logger         logging.Function
	tickets        ticket.Store
	sourceLogic    discovery.SourceLogic
	onScopeRequest scopeSource
	stats          stats
	mutex          sync.Mutex
}

func NewReporter(
	src session.ClientFactory,
	hostName *url.URL,
	supported stream.AddressSourceMap,
	tickets ticket.Store,
	onScopeRequest scopeSource,
	onAuth stream.AuthSource,
	metrics metrics.Tracker,
	logger logging.Function,
) (discovery.HostReporter, error) {
	if hostName == nil {
		return nil, protocol.ErrInvalidAddress
	}
	r := &reporter{
		onScopeRequest: onScopeRequest,
		supported:      supported,
		tickets:        tickets,
		status:         unsafe.Pointer(&message.StatusUnknown),
		logger:         logger,
		stats: stats{
			scopes:  metrics.Aggregator("host.scopes", []string{}),
			reports: metrics.Count("host.reports", []string{}),
			load:    metrics.Gauge("host.load", []string{}),
		},
	}
	initFunc, errChan := initializer(r)
	r.mutex.Lock()
	r.binding = newBinding(hostName.String(), src, r.onAllocate, r.onDetach, r.onReserve, initFunc, onAuth, logger)
	r.mutex.Unlock()
	err := <-errChan // block for authorization result
	return r, err
}

func initializer(r *reporter) (initFunction, chan error) {
	errChan := make(chan error)
	var once sync.Once
	return func(err error) {
		// report error to constructor on first attempt only
		once.Do(func() {
			if err != nil {
				errChan <- err
			}
			close(errChan)
		})
		// broker retry logic handles failure, ww only need
		// to process success here
		if err == nil {
			r.onInit()
		}
	}, errChan
}

func (r *reporter) Tick() {
	r.binding.tick()
}

func (r *reporter) Close() error {
	status := message.StatusClosed
	atomic.StorePointer(&r.status, unsafe.Pointer(&status))
	r.binding.send(status)
	r.binding.close(nil)
	return nil
}

func (r *reporter) Redeem(method stream.AuthMethod, code stream.OpaqueBytes) stream.CredentialsPromise {
	msg, err := message.NewValidate(r.binding.nextAckID(), code)
	if err != nil {
		r.logger(logging.Debug, err)
		return stream.NewSyncCredentialsPromise(nil, err)
	}
	return r.binding.validate(msg)
}

func (r *reporter) SetSourceLogic(sourceLogic discovery.SourceLogic) {
	r.mutex.Lock()
	r.sourceLogic = sourceLogic
	r.mutex.Unlock()
}

func (r *reporter) AddSupported(scopes stream.AddressSourceMap) {
	diff := int64(0)
	r.mutex.Lock()
	for k, v := range scopes {
		if _, found := r.supported[k]; !found {
			diff++
		}
		r.supported[k] = v
	}
	r.mutex.Unlock()
	r.stats.scopes.Add(diff)
	r.sendScopes(scopes, false)
}

func (r *reporter) DropSupported(scopes stream.AddressSourceMap) {
	// we should be able to heavily optimize this
	diff := int64(0)
	r.mutex.Lock()
	for k, v := range scopes {
		if held, found := r.supported[k]; found && held == v {
			delete(r.supported, k)
			diff--
		}
	}
	r.mutex.Unlock()
	r.stats.scopes.Add(diff)
	r.sendScopes(scopes, true)
}

func (r *reporter) UpdateStatus(load protocol.LoadFactor, flags protocol.StatusFlags) {
	status, _ := message.NewStatus(load, flags)
	atomic.StorePointer(&r.status, unsafe.Pointer(&status))
	r.stats.load.Set(float64(load))
	r.stats.reports.Add(1)
	r.binding.send(status) // TODO : retry on this send
}

func (r *reporter) getStatus() message.Status {
	return *(*message.Status)(atomic.LoadPointer(&r.status))
}

func (r *reporter) sendScopes(scopes stream.AddressSourceMap, remove bool) {
	msg, err := message.NewScopes(r.binding.nextAckID(), scopes, remove)
	if err != nil {
		r.logger(logging.Debug, err)
	} else {
		r.binding.send(msg)
	}
}

func (r *reporter) onInit() {
	r.mutex.Lock()
	scopes := r.supported
	r.mutex.Unlock()
	r.sendScopes(scopes, false)
	r.binding.send(r.getStatus())
	r.binding.finishInit()
}

func (r *reporter) onAllocate(req message.Allocate) {
	var msg protocol.Message
	var scopes stream.AddressScopes
	var err error

	r.mutex.Lock()
	sourceLogic := r.sourceLogic
	r.mutex.Unlock()

	flags := r.getStatus().Flags()
	if !flags.IsAvailable() || !flags.IsSource() || sourceLogic == nil {
		err = protocol.ErrServiceUnavailable
	} else {
		scopes, err = sourceLogic.ServeAddress(req.Address())
	}

	if err == nil {
		msg, err = message.NewAllocation(req.RequestID(), scopes)
	}

	if err != nil {
		msg, _ = message.NewFailure(req.RequestID(), err)
		r.logger(logging.Debug, "Allocate attempt failed", err)
	}
	r.binding.send(msg)
}

func (r *reporter) onDetach(req message.Detach) {
	var msg protocol.Message
	var scopes stream.AddressScopes
	var err error

	r.mutex.Lock()
	sourceLogic := r.sourceLogic
	r.mutex.Unlock()

	flags := r.getStatus().Flags()
	if !flags.IsAvailable() || !flags.IsSource() || sourceLogic == nil {
		err = protocol.ErrServiceUnavailable
	} else {
		scopes, err = sourceLogic.RevokeAddress(req.Address())
	}

	if err == nil {
		msg, err = message.NewDetached(req.RequestID(), scopes)
	}

	if err != nil {
		msg, _ = message.NewFailure(req.RequestID(), err)
		r.logger(logging.Debug, "Detach attempt failed", err)
	}
	r.binding.send(msg)
}

func (r *reporter) onReserve(req message.Reserve) {
	var scopes stream.AddressScopes
	var ticket stream.OpaqueBytes
	var err error

	creds := req.Credentials()
	addr := req.Address()
	if !r.getStatus().Flags().IsAvailable() {
		err = protocol.ErrServiceUnavailable
	} else {
		scopes, err = r.onScopeRequest(addr, creds)
		if err == nil {
			ticket, err = r.tickets.Request(creds)
		}
	}

	var msg protocol.Message
	reqID := req.RequestID()

	if err == nil {
		msg, err = message.NewTicket(reqID, ticket, scopes)
	}

	if err != nil {
		msg, _ = message.NewFailure(reqID, err)
		r.logger(logging.Debug, "Ticket attempt failed", err)
	}

	if msg != nil {
		r.binding.send(msg)
	}
}
