package protocol

import (
	"fmt"
	"net/http"

	"code.justin.tv/devhub/e2ml/libs/errors"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

const (
	ErrEOFCode                 = "eof"
	ErrInvalidAckIDCode        = "invalid_ack_id"
	ErrInvalidAddressCode      = "invalid_address"
	ErrInvalidBodyCode         = "invalid_body"
	ErrInvalidCredentialsCode  = "invalid_credentials"
	ErrInvalidHeaderCode       = "invalid_header"
	ErrInvalidHostCode         = "invalid_host"
	ErrInvalidLengthCode       = "invalid_length"
	ErrInvalidOpCodeCode       = "invalid_op_code"
	ErrInvalidPayloadCode      = "invalid_payload"
	ErrInvalidSegmentCode      = "invalid_segment"
	ErrInvalidRequestIDCode    = "invalid_request_id"
	ErrInvalidSuggestionIDCode = "invalid_suggestion_id"
	ErrInvalidVersionCode      = "invalid_version"
	ErrNoHostAvailableCode     = "no_host_available"
	ErrInvalidCacheEntryCode   = "invalid_cache_entry"
	ErrMissingCredentialsCode  = "missing_credentials"
	ErrServiceShuttingDownCode = "service_shutting_down"
	ErrServiceTimedOutCode     = "service_timed_out"
	ErrServiceUnavailableCode  = "service_unavailable"
)

type commonErrorSupport interface {
	error
	errors.ErrorCodeSource
	errors.DetailsSource
	errors.HTTPStatusSource
}

type errInvalidBody struct {
	inner error
}

func ErrInvalidBody(err error) error { return &errInvalidBody{err} }

var lookup *errors.Dictionary // late bound to Errors below
func ErrInvalidBodyBuilder(d errors.Details) error {
	reason := d.String("reason")
	var inner error
	if err := (*lookup).Unmarshal([]byte(reason), &inner); err != nil {
		inner = errors.NewBuilder(d.String("reason")).Build()
	}
	return ErrInvalidBody(inner)
}
func (e *errInvalidBody) Details() errors.Details {
	inner, err := Errors.Marshal(e.inner)
	if err != nil {
		inner = []byte(e.inner.Error())
	}
	return errors.Details{"reason": string(inner)}
}
func (*errInvalidBody) HTTPStatus() int   { return http.StatusInternalServerError }
func (*errInvalidBody) ErrorCode() string { return ErrInvalidBodyCode }
func (e *errInvalidBody) Error() string {
	return fmt.Sprintf("Invalid response body: %v", e.inner)
}

type errInvalidLength struct {
	expected int
	actual   int
}

func ErrInvalidLength(expected, actual int) error {
	return &errInvalidLength{expected, actual}
}
func ErrInvalidLengthBuilder(d errors.Details) error {
	return &errInvalidLength{int(d.Int64("expected")), int(d.Int64("actual"))}
}
func (e *errInvalidLength) HTTPStatus() int   { return http.StatusBadRequest }
func (e *errInvalidLength) ErrorCode() string { return ErrInvalidLengthCode }
func (e *errInvalidLength) Details() errors.Details {
	return errors.Details{"expected": e.expected, "actual": e.actual}
}
func (e *errInvalidLength) Error() string {
	return fmt.Sprintf("Invalid length (%d < %d)", e.actual, e.expected)
}

type errInvalidPayload struct {
	code OpCode
}

var _ commonErrorSupport = (*errInvalidPayload)(nil)

func ErrInvalidPayload(code OpCode) error {
	return &errInvalidPayload{code}
}
func ErrInvalidPayloadBuilder(d errors.Details) error {
	return ErrInvalidPayload(OpCode(d.Int64("code")))
}
func (e *errInvalidPayload) HTTPStatus() int         { return http.StatusBadRequest }
func (e *errInvalidPayload) ErrorCode() string       { return ErrInvalidPayloadCode }
func (e *errInvalidPayload) Details() errors.Details { return errors.Details{"code": int(e.code)} }
func (e *errInvalidPayload) Error() string {
	return fmt.Sprintf("Invalid payload for OpCode '%v'", e.code)
}

var internalList = []errors.ErrorCodeError{}

func quickBuild(code string, status int, msg string) errors.ErrorCodeError {
	err := errors.NewBuilder(msg).WithHTTPStatus(status).Build().WithErrorCode(code)
	internalList = append(internalList, err)
	return err
}

var (
	ErrEOF                 = quickBuild(ErrEOFCode, http.StatusUnprocessableEntity, "Message truncated")
	ErrForbidden           = stream.ErrForbidden
	ErrInvalidAckID        = quickBuild(ErrInvalidAckIDCode, http.StatusUnprocessableEntity, "Invalid ack id")
	ErrInvalidAddress      = quickBuild(ErrInvalidAddressCode, http.StatusUnprocessableEntity, "Invalid address")
	ErrInvalidHeader       = quickBuild(ErrInvalidHeaderCode, http.StatusBadRequest, "Invalid header")
	ErrInvalidHost         = quickBuild(ErrInvalidHostCode, http.StatusUnprocessableEntity, "Invalid host")
	ErrInvalidOpCode       = quickBuild(ErrInvalidOpCodeCode, http.StatusBadRequest, "Invalid op code")
	ErrInvalidSegment      = quickBuild(ErrInvalidSegmentCode, http.StatusUnprocessableEntity, "Invalid segment")
	ErrInvalidRequestID    = quickBuild(ErrInvalidRequestIDCode, http.StatusUnprocessableEntity, "Invalid request id")
	ErrInvalidSuggestionID = quickBuild(ErrInvalidRequestIDCode, http.StatusUnprocessableEntity, "Invalid suggestion id")
	ErrInvalidVersion      = quickBuild(ErrInvalidVersionCode, http.StatusBadRequest, "Invalid version")
	ErrNoHostAvailable     = quickBuild(ErrNoHostAvailableCode, http.StatusNotFound, "No host available")
	ErrInvalidCacheEntry   = quickBuild(ErrInvalidCacheEntryCode, http.StatusServiceUnavailable, "Invalid cache entry")
	ErrInvalidCredentials  = quickBuild(ErrInvalidCredentialsCode, http.StatusForbidden, "Invalid Credentials")
	ErrMissingCredentials  = quickBuild(ErrMissingCredentialsCode, http.StatusUnauthorized, "Missing Credentials")
	ErrServiceUnavailable  = quickBuild(ErrServiceUnavailableCode, http.StatusServiceUnavailable, "Service unavailable")
	ErrServiceTimedOut     = quickBuild(ErrServiceTimedOutCode, http.StatusServiceUnavailable, "Service timed out")
	ErrServiceShuttingDown = quickBuild(ErrServiceShuttingDownCode, http.StatusServiceUnavailable, "Service shutting down")

	Errors = func() errors.Dictionary {
		var builder = errors.NewDictionaryBuilder()
		builder.IncludeAll(stream.Errors)
		builder.Include(internalList...)
		builder.Map(ErrInvalidLengthCode, ErrInvalidLengthBuilder)
		builder.Map(ErrInvalidPayloadCode, ErrInvalidPayloadBuilder)
		builder.Map(ErrInvalidBodyCode, ErrInvalidBodyBuilder)
		return builder.Build()
	}()
)

func init() {
	// to enable recursive build of ErrInvalidBodyBuilder after initialization of Errors
	lookup = &Errors
}
