package protocol

import (
	"encoding/json"
	"net/http"
	"testing"

	"code.justin.tv/devhub/e2ml/libs/errors"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
)

type illegalError struct{}

var _ errors.DetailsError = (*illegalError)(nil)
var _ json.Marshaler = (*illegalError)(nil)

func (*illegalError) Error() string                  { return "boom" }
func (i *illegalError) Details() errors.Details      { return errors.Details{"k": i} }
func (i *illegalError) MarshalJSON() ([]byte, error) { return nil, i }

func TestErrInvalidBody(t *testing.T) {
	assert.EqualError(t, ErrInvalidBody(ErrForbidden), "Invalid response body: Access forbidden")
	assert.Equal(t, ErrInvalidBodyCode, errors.GetErrorCode(ErrInvalidBody(ErrForbidden)))
	assert.Equal(t, http.StatusInternalServerError, errors.GetHTTPStatusOrDefault(ErrInvalidBody(ErrForbidden), 0))
	assert.Equal(t, errors.Details{"reason": "{\"error_code\":\"access_forbidden\"}"}, errors.GetDetails(ErrInvalidBody(ErrForbidden)))
	assert.Equal(t, errors.Details{"reason": "boom"}, errors.GetDetails(ErrInvalidBody(&illegalError{})))
}

func TestErrInvalidLength(t *testing.T) {
	assert.EqualError(t, ErrInvalidLength(4, 0), "Invalid length (0 < 4)")
	assert.Equal(t, ErrInvalidLengthCode, errors.GetErrorCode(ErrInvalidLength(0, 0)))
	assert.Equal(t, http.StatusBadRequest, errors.GetHTTPStatusOrDefault(ErrInvalidLength(0, 0), 0))
	assert.Equal(t, errors.Details{"expected": 4, "actual": 0}, errors.GetDetails(ErrInvalidLength(4, 0)))
}

func TestErrInvalidPayload(t *testing.T) {
	assert.EqualError(t, ErrInvalidPayload(AuthHost), "Invalid payload for OpCode 'auth_host'")
	assert.Equal(t, ErrInvalidPayloadCode, errors.GetErrorCode(ErrInvalidPayload(AuthHost)))
	assert.Equal(t, http.StatusBadRequest, errors.GetHTTPStatusOrDefault(ErrInvalidPayload(AuthHost), 0))
	assert.Equal(t, errors.Details{"code": int(AuthHost)}, errors.GetDetails(ErrInvalidPayload(AuthHost)))
}

func TestErrors(t *testing.T) {
	assert.Equal(t, stream.ErrInvalidNamespace, Errors.Get(stream.ErrInvalidNamespaceCode, errors.Details{}))
	assert.Equal(t, ErrEOF, Errors.Get(ErrEOFCode, errors.Details{}))
	assert.Equal(t, ErrInvalidLength(10, 20), Errors.Get(ErrInvalidLengthCode, errors.Details{"expected": 10, "actual": 20}))
	assert.Equal(t, ErrInvalidPayload(AuthHost), Errors.Get(ErrInvalidPayloadCode, errors.Details{"code": int(AuthHost)}))
	assert.Equal(t, ErrInvalidBody(ErrEOF), Errors.Get(ErrInvalidBodyCode, errors.Details{"reason": "{\"error_code\":\"eof\"}"}))
	assert.Equal(t, ErrInvalidBody(errors.NewBuilder("boom").Build()), Errors.Get(ErrInvalidBodyCode, errors.Details{"reason": "boom"}))
}
