package message

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestAccept(t *testing.T) {
	sid := protocol.FirstSuggestion("source")
	host := "wss://localhost:8080"
	addr, errx := stream.NewAddress(stream.Namespace("test"), stream.Version(1), map[string]string{})
	require.NoError(t, errx)

	t.Run("should report values correctly", func(t *testing.T) {
		request, err := NewAccept(addr, sid, host)
		require.NoError(t, err)
		assert.Equal(t, protocol.Accept, request.OpCode())
		assert.Equal(t, sid, request.ID())
		assert.Equal(t, addr, request.Address())
		assert.Equal(t, host, request.Hostname())
		assert.Equal(t, "<accept> addr=test@1, suggestion=source#0, hostname=wss://localhost:8080", request.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		request, err := NewAccept(addr, sid, host)
		require.NoError(t, err)
		bytes, err := request.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, request, out)
	})

	t.Run("should report suggestion id errors", func(t *testing.T) {
		_, err := NewAccept(addr, protocol.NoSuggestion, host)
		assert.Equal(t, protocol.ErrInvalidSuggestionID, err)
		_, err = (&acceptMessage{addr, protocol.NoSuggestion, host}).Marshal(protocol.Current)
		assert.Equal(t, protocol.ErrInvalidSuggestionID, err)
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Accept).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Accept).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report address errors", func(t *testing.T) {
		_, err := NewAccept(nil, sid, host)
		assert.Equal(t, protocol.ErrInvalidAddress, err)
		bytes, err := (&acceptMessage{&badAddr{}, sid, host}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, stream.ErrMissingRequiredVersion, createBlank(protocol.Accept).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report empty hostname", func(t *testing.T) {
		_, err := NewAccept(addr, sid, "")
		assert.Equal(t, protocol.ErrInvalidHost, err)
		bytes, err := (&acceptMessage{addr, sid, ""}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, protocol.ErrInvalidHost, createBlank(protocol.Accept).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report invalid hostname", func(t *testing.T) {
		_, err := NewAccept(addr, sid, "bad!url:")
		assert.Equal(t, protocol.ErrInvalidHost, err)
		bytes, err := (&acceptMessage{addr, sid, "bad!url:"}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, protocol.ErrInvalidHost, createBlank(protocol.Accept).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(acceptAddressOffset, 0),
			createBlank(protocol.Accept).Unmarshal(protocol.Current, []byte{}))
	})

	t.Run("should handle EOF gracefully", func(t *testing.T) {
		broken := make([]byte, acceptAddressOffset)
		injectHeader(broken, protocol.One, protocol.Accept)
		broken = append(broken, "key"...)
		assert.Equal(t, protocol.ErrEOF, createBlank(protocol.Accept).Unmarshal(protocol.Current, broken))
		broken = append(broken, "\000source"...)
		assert.Equal(t, protocol.ErrEOF, createBlank(protocol.Accept).Unmarshal(protocol.Current, broken))
	})
}
