package message

import (
	"fmt"
	"net/url"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

const (
	acceptedAddressOffset = headerLength
)

// Accepted forwards a proposal after getting Promise from a quorum
type Accepted interface {
	protocol.Message
	protocol.Proposal
}

type acceptedMessage struct {
	addr stream.Address
	id   protocol.SuggestionID
	host string
}

func NewAccepted(addr stream.Address, id protocol.SuggestionID, hostname string) (Accepted, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidSuggestionID
	}
	if addr == nil {
		return nil, protocol.ErrInvalidAddress
	}
	if hostname == "" {
		return nil, protocol.ErrInvalidHost
	}
	if _, err := url.Parse(hostname); err != nil {
		return nil, protocol.ErrInvalidHost
	}
	return &acceptedMessage{addr, id, hostname}, nil
}

func (*acceptedMessage) OpCode() protocol.OpCode     { return protocol.Accepted }
func (a *acceptedMessage) ID() protocol.SuggestionID { return a.id }
func (a *acceptedMessage) Address() stream.Address   { return a.addr }
func (a *acceptedMessage) Hostname() string          { return a.host }

func (a *acceptedMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	if !a.id.IsValid() {
		return nil, protocol.ErrInvalidSuggestionID
	}
	key := stream.NoAddress
	if a.addr != nil {
		key = a.addr.Key()
	}
	suggestionOffset := acceptedAddressOffset + len(key)
	hostOffset := suggestionOffset + 1 + a.id.Len()
	bytes := make([]byte, hostOffset+1+len(a.host))
	injectHeader(bytes, ver, a.OpCode())
	copy(bytes[acceptAddressOffset:], key)
	bytes[suggestionOffset] = '\000'
	err := a.id.Marshal(bytes[suggestionOffset+1:])
	bytes[hostOffset] = '\000'
	copy(bytes[hostOffset+1:], a.host)
	return bytes, err
}

func (a *acceptedMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	total := len(bytes)
	if total < acceptAddressOffset {
		return protocol.ErrInvalidLength(acceptAddressOffset, total)
	}

	suggestionOffset := acceptAddressOffset
	for suggestionOffset < total {
		if bytes[suggestionOffset] == '\000' {
			break
		}
		suggestionOffset++
	}
	if suggestionOffset == total {
		return protocol.ErrEOF
	}

	hostOffset := suggestionOffset + 5
	for hostOffset < total {
		if bytes[hostOffset] == '\000' {
			break
		}
		hostOffset++
	}
	if hostOffset >= total {
		return protocol.ErrEOF
	}

	addr, err := stream.ParseAddress(string(bytes[acceptAddressOffset:suggestionOffset]))
	if err != nil {
		return err
	}

	hostname := string(bytes[hostOffset+1:])
	host, err := url.Parse(hostname)
	if err != nil || hostname == "" {
		return protocol.ErrInvalidHost
	}
	a.addr = addr
	a.id, _ = protocol.UnmarshalSuggestion(bytes[suggestionOffset+1 : hostOffset])
	a.host = host.String()
	return nil
}

func (a *acceptedMessage) String() string {
	return fmt.Sprintf("<%v> suggestion=%v, addr=%s, hostname=%s", a.OpCode(), a.ID(), a.Address().Key(), a.Hostname())
}
