package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
)

const (
	ackLength      = 6
	ackAckIDOffset = headerLength
)

type Ack interface {
	protocol.Message
	ForAckID() protocol.AckID
}

type ackMessage struct {
	forId protocol.AckID
}

// NewAck creates an Ack
func NewAck(id protocol.AckID) (Ack, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidAckID
	}
	return &ackMessage{id}, nil
}

func (*ackMessage) OpCode() protocol.OpCode    { return protocol.Ack }
func (a *ackMessage) ForAckID() protocol.AckID { return a.forId }
func (a *ackMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	bytes := make([]byte, ackLength)
	injectHeader(bytes, ver, a.OpCode())
	binary.BigEndian.PutUint16(bytes[ackAckIDOffset:], uint16(a.forId))
	return bytes, nil
}

func (a *ackMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	if len(bytes) != ackLength {
		return protocol.ErrInvalidLength(ackLength, len(bytes))
	}

	id := protocol.AckID(binary.BigEndian.Uint16(bytes[ackAckIDOffset:]))
	if !id.IsValid() {
		return protocol.ErrInvalidAckID
	}
	a.forId = id
	return nil
}

func (a *ackMessage) String() string {
	return fmt.Sprintf("<%v> reqID=%d", a.OpCode(), a.ForAckID())
}
