package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

const (
	allocateRequestIDOffset = headerLength
	allocateAddressOffset   = allocateRequestIDOffset + 4
)

// Allocate attempts to reserve a connection slot
type Allocate interface {
	protocol.Request
	Address() stream.Address
}

type allocateMessage struct {
	id   protocol.RequestID
	addr stream.Address
}

// NewAllocate creates a Allocate
func NewAllocate(id protocol.RequestID, addr stream.Address) (Allocate, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidRequestID
	}
	if addr == nil {
		return nil, protocol.ErrInvalidAddress
	}
	return &allocateMessage{id, addr}, nil
}

func (*allocateMessage) OpCode() protocol.OpCode         { return protocol.Allocate }
func (a *allocateMessage) RequestID() protocol.RequestID { return a.id }
func (a *allocateMessage) Address() stream.Address       { return a.addr }

func (a *allocateMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	key := stream.NoAddress
	if a.addr != nil {
		key = a.addr.Key()
	}
	bytes := make([]byte, allocateAddressOffset+len(key))
	injectHeader(bytes, ver, a.OpCode())
	binary.BigEndian.PutUint32(bytes[allocateRequestIDOffset:], uint32(a.id))
	copy(bytes[allocateAddressOffset:], key)

	return bytes, nil
}

func (a *allocateMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	total := len(bytes)
	if total < allocateAddressOffset {
		return protocol.ErrInvalidLength(allocateAddressOffset, total)
	}

	id := protocol.RequestID(binary.BigEndian.Uint32(bytes[allocateRequestIDOffset:]))
	if !id.IsValid() {
		return protocol.ErrInvalidRequestID
	}
	addr, err := stream.ParseAddress(string(bytes[allocateAddressOffset:]))
	if err != nil {
		return err
	}
	a.id = id
	a.addr = addr
	return nil
}

func (a *allocateMessage) String() string {
	return fmt.Sprintf("<%v> reqID=%d, addr=%s", a.OpCode(), a.RequestID(), a.Address().Key())
}
