package message

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestAllocate(t *testing.T) {
	addr, errx := stream.NewAddress(stream.Namespace("test"), stream.Version(1), map[string]string{})
	require.NoError(t, errx)

	t.Run("should report values correctly", func(t *testing.T) {
		allocate, err := NewAllocate(protocol.FirstRequestID, addr)
		require.NoError(t, err)
		assert.Equal(t, protocol.Allocate, allocate.OpCode())
		assert.Equal(t, protocol.FirstRequestID, allocate.RequestID())
		assert.Equal(t, addr, allocate.Address())
		assert.Equal(t, "<allocate> reqID=1, addr=test@1", allocate.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		allocate, err := NewAllocate(protocol.FirstRequestID, addr)
		require.NoError(t, err)
		bytes, err := allocate.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, allocate, out)
	})

	t.Run("should report allocate id errors", func(t *testing.T) {
		_, err := NewAllocate(protocol.NoRequestID, addr)
		assert.Equal(t, protocol.ErrInvalidRequestID, err)
		bytes, err := (&allocateMessage{protocol.NoRequestID, addr}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, protocol.ErrInvalidRequestID, createBlank(protocol.Allocate).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Allocate).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Allocate).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report address errors", func(t *testing.T) {
		_, err := NewAllocate(protocol.FirstRequestID, nil)
		assert.Equal(t, protocol.ErrInvalidAddress, err)
		bytes, err := (&allocateMessage{protocol.FirstRequestID, &badAddr{}}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, stream.ErrMissingRequiredVersion, createBlank(protocol.Allocate).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(allocateAddressOffset, 0),
			createBlank(protocol.Allocate).Unmarshal(protocol.Current, []byte{}))
	})
}
