package message

import (
	"encoding/binary"
	"fmt"
	"sort"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

const (
	allocationRequestIDOffset = headerLength
	allocationScopesOffset    = allocationRequestIDOffset + 4
)

// TODO : report source ids to allow consistent and fair detach logic
// Allocation reports a successful allocate attempt
type Allocation interface {
	protocol.Response
	Scopes() stream.AddressScopes
}

type allocationMessage struct {
	id     protocol.RequestID
	scopes stream.AddressScopes
}

// NewAllocation creates an Allocation
func NewAllocation(id protocol.RequestID, scopes stream.AddressScopes) (Allocation, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidRequestID
	}
	sorted := make(stream.AddressScopes, len(scopes))
	copy(sorted, scopes)
	sort.Sort(sorted)
	return &allocationMessage{id, sorted}, nil
}

func (*allocationMessage) OpCode() protocol.OpCode            { return protocol.Allocation }
func (a *allocationMessage) ForRequestID() protocol.RequestID { return a.id }
func (a *allocationMessage) Scopes() stream.AddressScopes     { return a.scopes }

func (a *allocationMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}

	data, _ := a.scopes.MarshalBinary()
	bytes := make([]byte, allocateAddressOffset+len(data))
	injectHeader(bytes, ver, a.OpCode())
	binary.BigEndian.PutUint32(bytes[allocationRequestIDOffset:], uint32(a.id))
	copy(bytes[allocationScopesOffset:], data)

	return bytes, nil
}

func (a *allocationMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	total := len(bytes)
	if total < allocationScopesOffset {
		return protocol.ErrInvalidLength(allocationScopesOffset, total)
	}

	id := protocol.RequestID(binary.BigEndian.Uint32(bytes[allocationRequestIDOffset:]))
	if !id.IsValid() {
		return protocol.ErrInvalidRequestID
	}
	var scopes stream.AddressScopes
	if err := scopes.UnmarshalBinary(bytes[allocationScopesOffset:]); err != nil {
		return err
	}
	a.id = id
	a.scopes = scopes
	return nil
}

func (a *allocationMessage) String() string {
	return fmt.Sprintf("<%v> reqID=%d, scopes=%s", a.OpCode(), a.ForRequestID(), a.Scopes())
}
