package message

import (
	"sort"
	"testing"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestAllocation(t *testing.T) {
	addr, errx := stream.NewAddress(stream.Namespace("test"), stream.Version(1), map[string]string{})
	require.NoError(t, errx)
	scopes := stream.AddressScopes{stream.AnyAddress, addr}

	t.Run("should report values correctly", func(t *testing.T) {
		allocation, err := NewAllocation(protocol.FirstRequestID, scopes)
		require.NoError(t, err)
		assert.Equal(t, protocol.Allocation, allocation.OpCode())
		assert.Equal(t, protocol.FirstRequestID, allocation.ForRequestID())
		assert.NotEqual(t, scopes, allocation.Scopes()) // due to resort
		sorted := make(stream.AddressScopes, len(scopes))
		copy(sorted, scopes)
		sort.Sort(sorted)
		assert.Equal(t, sorted, allocation.Scopes())
		assert.Equal(t, "<allocation> reqID=1, scopes=[^test@1 ^*]", allocation.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		allocation, err := NewAllocation(protocol.FirstRequestID, scopes)
		require.NoError(t, err)
		bytes, err := allocation.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, allocation, out)
	})

	t.Run("should report allocation id errors", func(t *testing.T) {
		_, err := NewAllocation(protocol.NoRequestID, scopes)
		assert.Equal(t, protocol.ErrInvalidRequestID, err)
		bytes, err := (&allocationMessage{protocol.NoRequestID, scopes}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, protocol.ErrInvalidRequestID, createBlank(protocol.Allocation).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Allocation).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Allocation).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report address errors", func(t *testing.T) {
		msg, _ := NewAllocation(protocol.FirstRequestID, stream.AddressScopes{&badAddr{}})
		bytes, err := msg.Marshal(protocol.Current)
		require.NoError(t, err) // marshal doesn't validate
		assert.Equal(t, stream.ErrMissingRequiredVersion, createBlank(protocol.Allocation).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(allocationScopesOffset, 0),
			createBlank(protocol.Allocation).Unmarshal(protocol.Current, []byte{}))
	})
}
