package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

const (
	authAckIDOffset    = headerLength
	authHostnameOffset = authAckIDOffset + 2
)

// AuthHost is used to establish an initial handshake between host and service
type AuthHost interface {
	protocol.Ackable
	Equals(AuthHost) bool
	Hostname() string
	Token() stream.OpaqueBytes
	Version() protocol.Version
}

type authHostMessage struct {
	id       protocol.AckID
	ver      protocol.Version
	hostname string
	token    stream.OpaqueBytes
}

// NewAuthHost creates an AuthHost
func NewAuthHost(id protocol.AckID, ver protocol.Version, hostname string, token OpaqueBytes) (AuthHost, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidAckID
	}
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	return &authHostMessage{id, ver, hostname, token}, nil
}

func (*authHostMessage) OpCode() protocol.OpCode     { return protocol.AuthHost }
func (a *authHostMessage) AckID() protocol.AckID     { return a.id }
func (a *authHostMessage) Version() protocol.Version { return a.ver }
func (a *authHostMessage) Hostname() string          { return a.hostname }
func (a *authHostMessage) Token() stream.OpaqueBytes { return a.token }

func (a *authHostMessage) Equals(other AuthHost) bool {
	return a.Hostname() == other.Hostname() && a.Version() == other.Version()
}

func (a *authHostMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	authTokenOffset := authHostnameOffset + len(a.hostname)
	bytes := make([]byte, authTokenOffset+1+len(a.token))
	injectHeader(bytes, ver, a.OpCode())
	binary.BigEndian.PutUint16(bytes[authAckIDOffset:], uint16(a.id))
	copy(bytes[authHostnameOffset:], a.hostname)
	bytes[authTokenOffset] = '\000'
	copy(bytes[authTokenOffset+1:], a.token)
	return bytes, nil
}

func (a *authHostMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	total := len(bytes)
	if total < authHostnameOffset {
		return protocol.ErrInvalidLength(authHostnameOffset, total)
	}

	authTokenOffset := authHostnameOffset
	for authTokenOffset < total {
		if bytes[authTokenOffset] == '\000' {
			break
		}
		authTokenOffset++
	}

	if authTokenOffset == total {
		return protocol.ErrEOF
	}

	id := protocol.AckID(binary.BigEndian.Uint16(bytes[authAckIDOffset:]))
	if !id.IsValid() {
		return protocol.ErrInvalidAckID
	}

	a.id = id
	a.ver = ver
	a.hostname = string(bytes[authHostnameOffset:authTokenOffset])
	a.token = OpaqueBytes(bytes[authTokenOffset+1:])
	return nil
}

func (a *authHostMessage) String() string {
	return fmt.Sprintf("<%v> reqID=%d, hostname=%s, version=%d, ####", a.OpCode(), a.AckID(), a.Hostname(), a.Version())
}
