package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

const (
	bindAckIDOffset = headerLength
	bindTokenOffset = bindAckIDOffset + 4
)

// Bind tells a host to use credentials for an token
type Bind interface {
	protocol.Message
	ForAckID() protocol.AckID
	Token() OpaqueBytes // remove any doubt around ackID collision
	Credentials() stream.Credentials
}

type bindMessage struct {
	id    protocol.AckID
	token OpaqueBytes
	creds stream.Credentials
}

// NewBind creates a Bind
func NewBind(id protocol.AckID, token OpaqueBytes, creds stream.Credentials) (Bind, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidAckID
	}
	if creds == nil {
		return nil, protocol.ErrMissingCredentials
	}
	return &bindMessage{id, token, creds}, nil
}

func (*bindMessage) OpCode() protocol.OpCode           { return protocol.Bind }
func (b *bindMessage) ForAckID() protocol.AckID        { return b.id }
func (b *bindMessage) Token() OpaqueBytes              { return b.token }
func (b *bindMessage) Credentials() stream.Credentials { return b.creds }

func (b *bindMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	cbytes, _ := b.creds.MarshalBinary()
	bytes := make([]byte, bindTokenOffset+len(b.token)+1+len(cbytes))
	injectHeader(bytes, ver, b.OpCode())
	binary.BigEndian.PutUint16(bytes[bindAckIDOffset:], uint16(b.id))
	copy(bytes[bindTokenOffset:], b.token)

	authOffset := bindTokenOffset + len(b.token)
	bytes[authOffset] = '\000'
	copy(bytes[authOffset+1:], cbytes)

	return bytes, nil
}

func (b *bindMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	total := len(bytes)
	if total < bindTokenOffset {
		return protocol.ErrInvalidLength(bindTokenOffset, total)
	}
	authOffset := bindTokenOffset
	for authOffset < total {
		if bytes[authOffset] == '\000' {
			break
		}
		authOffset++
	}
	if authOffset == total {
		return protocol.ErrEOF
	}

	id := protocol.AckID(binary.BigEndian.Uint16(bytes[bindAckIDOffset:]))
	if !id.IsValid() {
		return protocol.ErrInvalidAckID
	}
	creds, err := stream.UnmarshalCredentials(bytes[authOffset+1:])
	if err != nil {
		return err
	}
	b.id = id
	b.token = OpaqueBytes(bytes[bindTokenOffset:authOffset])
	b.creds = creds
	return nil
}

func (b *bindMessage) String() string {
	return fmt.Sprintf("<%v> reqID=%d, ####, creds=%s", b.OpCode(), b.ForAckID(), b.Credentials().ClientID())
}
