package message

import (
	"testing"
	"time"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestBind(t *testing.T) {
	token := OpaqueBytes("NotATicket")
	creds := stream.NewTimedCredentials(
		"client",
		stream.AddressScopes{stream.AnyAddress},
		stream.AddressScopes{},
		time.Now().AddDate(0, 0, 1),
	)

	t.Run("should report values correctly", func(t *testing.T) {
		bind, err := NewBind(protocol.FirstAckID, token, creds)
		require.NoError(t, err)
		assert.Equal(t, protocol.Bind, bind.OpCode())
		assert.Equal(t, protocol.FirstAckID, bind.ForAckID())
		assert.Equal(t, token, bind.Token())
		assert.Equal(t, creds, bind.Credentials())
		assert.Equal(t, "<bind> reqID=1, ####, creds=client", bind.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		bind, err := NewBind(protocol.FirstAckID, token, creds)
		require.NoError(t, err)
		bytes, err := bind.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, bind, out)
	})

	t.Run("should report ack id errors", func(t *testing.T) {
		_, err := NewBind(protocol.NoAckID, token, creds)
		assert.Equal(t, protocol.ErrInvalidAckID, err)
		bytes, err := (&bindMessage{protocol.NoAckID, token, creds}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, protocol.ErrInvalidAckID, createBlank(protocol.Bind).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Bind).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Bind).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report credential errors", func(t *testing.T) {
		_, err := NewBind(protocol.FirstAckID, token, nil)
		assert.Equal(t, protocol.ErrMissingCredentials, err)
		bytes, err := (&bindMessage{protocol.FirstAckID, token, &badCreds{}}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, stream.ErrInvalidCredentialFormat, createBlank(protocol.Bind).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(bindTokenOffset, 0),
			createBlank(protocol.Bind).Unmarshal(protocol.Current, []byte{}))
	})

	t.Run("should handle EOF gracefully", func(t *testing.T) {
		broken := make([]byte, bindTokenOffset)
		injectHeader(broken, protocol.One, protocol.Bind)
		broken = append(broken, "key"...)
		assert.Equal(t, protocol.ErrEOF, createBlank(protocol.Bind).Unmarshal(protocol.Current, broken))
	})
}
