package message

import (
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

const (
	connectTokenOffset = headerLength
)

// Connect establishes the handshake for a peer-to-peer connection; it must be
// the first message sent in both directions. Once a connect message has been
// verified, the peered host will send initial state for all local reporters
// followed by a Ready message that means it will begin normal operation over
// the link
type Connect interface {
	protocol.Message
	Version() protocol.Version
	Token() OpaqueBytes
}

type connectMessage struct {
	token stream.OpaqueBytes
	ver   protocol.Version
}

func NewConnect(ver protocol.Version, token OpaqueBytes) (Connect, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	return &connectMessage{token, ver}, nil
}

func (*connectMessage) OpCode() protocol.OpCode     { return protocol.Connect }
func (c *connectMessage) Version() protocol.Version { return c.ver }
func (c *connectMessage) Token() OpaqueBytes        { return c.token }

func (c *connectMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	bytes := make([]byte, connectTokenOffset+len(c.token))
	injectHeader(bytes, ver, c.OpCode())
	copy(bytes[connectTokenOffset:], c.token)
	return bytes, nil
}

func (c *connectMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	total := len(bytes)
	if total < connectTokenOffset {
		return protocol.ErrInvalidLength(connectTokenOffset, total)
	}
	c.ver = ver
	c.token = OpaqueBytes(bytes[connectTokenOffset:])
	return nil
}

func (c *connectMessage) String() string {
	return fmt.Sprintf("<%v> version=%v, ####", c.OpCode(), c.Version())
}
