package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/discovery/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

const (
	detachRequestIDOffset = headerLength
	detachAddressOffset   = detachRequestIDOffset + 4
)

// Detach attempts to reserve a connection slot
type Detach interface {
	protocol.Request
	Address() stream.Address
}

type detachMessage struct {
	id   protocol.RequestID
	addr stream.Address
}

// NewDetach creates a Detach
func NewDetach(id protocol.RequestID, addr stream.Address) (Detach, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidRequestID
	}
	if addr == nil {
		return nil, protocol.ErrInvalidAddress
	}
	return &detachMessage{id, addr}, nil
}

func (*detachMessage) OpCode() protocol.OpCode         { return protocol.Detach }
func (d *detachMessage) RequestID() protocol.RequestID { return d.id }
func (d *detachMessage) Address() stream.Address       { return d.addr }

func (d *detachMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if ver != protocol.Two {
		return nil, protocol.ErrInvalidVersion
	}
	key := stream.NoAddress
	if d.addr != nil {
		key = d.addr.Key()
	}
	bytes := make([]byte, detachAddressOffset+len(key))
	injectHeader(bytes, ver, d.OpCode())
	binary.BigEndian.PutUint32(bytes[detachRequestIDOffset:], uint32(d.id))
	copy(bytes[detachAddressOffset:], key)

	return bytes, nil
}

func (d *detachMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if ver != protocol.Two {
		return protocol.ErrInvalidVersion
	}
	total := len(bytes)
	if total < detachAddressOffset {
		return protocol.ErrInvalidLength(detachAddressOffset, total)
	}

	id := protocol.RequestID(binary.BigEndian.Uint32(bytes[detachRequestIDOffset:]))
	if !id.IsValid() {
		return protocol.ErrInvalidRequestID
	}
	addr, err := stream.ParseAddress(string(bytes[detachAddressOffset:]))
	if err != nil {
		return err
	}
	d.id = id
	d.addr = addr
	return nil
}

func (d *detachMessage) String() string {
	return fmt.Sprintf("<%v> reqID=%d, addr=%s", d.OpCode(), d.RequestID(), d.Address().Key())
}
